/* SIMILARITY REDUCTION OF A GENERAL MATRIX TO HESSENBERG FORM             */
/*                                                                         */
/* R. S. Martin and J. H. Wilkinson.                                       */

/* From J. H. Wilkinson & C. Reinsch, "Handbook for Automatic Computation",
   Vol. II, "Linear Algebra", Springer-Verlag, New York, 1971, pp. 344-351. */

/* P. 344 */

/* Given the unsymmetric matrix, A, stored in the array a[1:n, 1:n],       */
/* this procedure reduces the sub-matrix of order l-k+1, which starts      */
/* at the element a[k, k] and finishes at the element a[l, l], to          */
/* Hessenberg form, H, by non-orthogonal elementary transformations.       */
/* The matrix H is overwritten on A with details of the transformations    */
/* stored in the remaining triangle under H and in the array int[k:l].     */

/* Tested with data on page 353 with the same results as on page 354.      */
/* 11/5/2008                                                               */

elmhes: procedure (n, k, l, a, int) options (reorder);
   DECLARE (n, k, l) FIXED BINARY (31);
   DECLARE a(*,*) FLOAT (18);
   DECLARE int (*) FIXED BINARY (31);

/* Input to procedure elmhes                                               */
/*    n      order of the full matrix A.                                   */
/*    k,l    parameters output by a procedure for preparing A.             */
/*           If A is not prepared in this way k = 1, l = n.                */
/*    a      the n x n matrix A, normally in prepared form [3].            */
/* Output from procedure elmhes                                            */
/*    a      an n x n array consisting partly of the derived upper         */
/*           Hessenberg matrix; the quantity n(i, r+1) involved in the     */
/*           reduction is stored in the [i, r] element.                    */

   DECLARE (i, j, la, m) FIXED BINARY (31);
   DECLARE (x, y) FLOAT (18);

   la = l - 1;
   do m = k + 1 to la;
      i=m; x=0;
      do j = m to l;
         if abs(a(j, m-1)) > abs(x) then
            do; x = a(j, m-1); i=j; end;
      end;
      int(m) = i;
      if i ^= m then
         do; /* interchange rows and columns of array a */
            do j = m-1 to n;
               y = a(i,j); a(i,j) = a(m,j); a(m,j) = y;
            end;
            do j = 1 to l;
               y = a(j,i); a(j,i) = a(j, m); a(j,m) = y;
            end;
         end; /* interchange */
      if x ^= 0 then
         do i = m+1 to l;
            y = a(i,m-1);
            if y ^= 0 then
               do;
                  y, a(i, m-1) = y/x;
                  do j = m to n;
                     a(i,j) = a(i,j) - y * a(m,j);
                  end;
                  do j = 1 to l;
                     a(j, m) = a(j, m) + y * a(j, i);
                  end;
               end;
         end;
   end;
end elmhes;

/* P. 345 */

/* Given r eigenvectors, stored as columns in the array z[1:n, 1:r], of
/* the Hessenberg matrix, H, which was formed by elmhes, and details
/* of the transformations as left by elmhes below H and in the array
/* int[k:l], 1 <= h <= l <= n, this procedure forms the eigenvectors
/* of the matrix A and overwrites them on the given vectors.               */

elmbak: PROCEDURE (k, l, r, a, int, z)  OPTIONS (REORDER);
   DECLARE (k, l, r) FIXED BINARY (31);
   DECLARE (a, z) (*,*) FLOAT (18);
   DECLARE int(*) FIXED BINARY (31);

/* Input to procedure elmbak
/*    k, I  parameters output by a procedure preparing A for reduction to
/*          Hessenberg form [3].
/*    r     the number of eigenvectors of the Hessenberg matrix which have
/*          been determined.
/*    a     the n x n array output by elmhes which includes the row and
/*          column multipliers used in the reduction.
/*    int   an integer array describing the row and column interchanges
/*          used in elmhes.
/*    z     an n x r array giving r eigenvectors of the derived Hessenberg
/*          matrix, corresponding to a real eigenvalue there is one column
/*          of this array; corresponding to a complex conjugate pair of
/*          eigenvalues lambda+- imu the matrix z contains two consecutive
/*          columns a and b such that a+-ib are the vectors corresponding
/*          to lambda+-imu respectively.
/* Output from procedure elmbak
/*    z     an n x r array giving r eigenvectors of the matrix reduced by
/*          elmhes iron analogous form to that used in the input z. The
/*          output vectors are not normalized since usually they will be
/*          the eigenvectors of the prepared version of the original
/*          matrix and will need further back transformation to give
/*          eigenvectors of the original matrix (see [3]). Moreover
/*          complex conjugate pairs would need another form of
/*          normalization.                                                 */

   DECLARE (i,j,m) FIXED BINARY (31);
   DECLARE x FLOAT (18);

   k = k + 1;
   do m = l-1 to k by -1;
      do i = m+1 to l;
         x = a(i,m-1);
         if x ^= 0 then
            do j = 1 to r;
               z(i,j) = z(i,j) + x*z(m,j);
            end;
         i = int(m);
         if i^=m then
            do j = 1 to r;
               x = z(i,j); z(i,j) = z(m,j); z(m,j) = x;
            end;
      end;
   end;
end elmbak;

/* P. 346 */

/* Given the unsymmetric matrix, A, stored in the array a[1:n, 1:n],       */
/* this procedure reduces the sub-matrix of order l-k+1, which starts      */
/* at the element a[k, k] and finishes at the element a[l, I], to          */
/* Hessenberg form, H, by the direct method (AN=NH). The matrix H is       */
/* overwritten on A with details of the transformations (N) stored in      */
/* the remaining triangle under H and in the array int[k:f]. The code      */
/* procedure innerprod is used. See Contribution I/2.                      */

dirhes: PROCEDURE (n, k, l, a, int) OPTIONS (REORDER);
   DECLARE (n, k, l) FIXED BINARY (31);
   DECLARE a(*,*) FLOAT (18), int(*) FIXED BINARY (31);

/* Input to procedure dirhes
/*    n     order of full matrix A
/*    k, l  parameters output by a procedure for preparing A. (See [3].)
/*          If A is not prepared in this way k = 1, l = n.
/*    a     the n x n matrix A, normally in prepared form [3].
/* Output from procedure dirhes.
/*    a     an n x n array consisting partly of the derived upper
/*          Hessenberg matrix ; the element n(i,r+1) of N is stored in
/*          the [i, r] position.
/*    int   an integer array describing the row and column interchanges
/*          involved in the reduction.                                     */

   DECLARE (i, ij, j, m) FIXED BINARY (31);
   DECLARE (x, y, z) FLOAT (18);

   do j = k+1 to l;
      m=j; x=0;
      do i = j to l;
         if abs(a(i,j-1)) > abs(x) then
            do;
               x = a(i,j-1); m=i;
            end;
         int(j) = m;
         if m ^= j then
            do;
               /* interchange rows and columns of a. */
               do i = k to n;
                  y = a(m,i); a(m,i) = a(j,i); a(j,i) = y;
               end;
               do i = 1 to l;
                  y = a(i,m); a(i,m) = a(i,j); a(i,j) = y;
               end;
            end; /* interchange */
         if x ^= 0 then
            do i = j+1 to l;
               a(i,j-1) = a(i,j-1) / x;
            end;
         do i = 1 to l;
            y = a(i,j);
            if x ^= 0 then
               call innerprod (j+1, 1, l, y, 0, a(i,m), a(m,j-1), m, y, z);
            else
               z= 0;
            if i <= j then ij = i-1; else ij = j;
            call innerprod (k+1, 1, ij, -y, -z, a(i,m-1), a(m,j), m, y, z);
            a(i,j) = -y;
         end;
      end;
   end;
end dirhes;

/* P. 347 */

/* Variable L added to declarations, 22/3/2007 */
/* Variable N was in error in line no. 80. */
/* LBN, UBN and N were deleted. */

/* Given r eigenvectors, stored as columns in the array z[1:n, 1:r], of    */
/* the Hessenberg matrix, H, which was formed by dirhes, and details       */
/* of the transformations as left by dirhes below H and in the array       */
/* int[k:l], 1 <= k <= l <= n, this procedure forms the eigenvectors       */
/* of the matrix A and overwrites them on the given vectors. The pro-      */
/* cedure innerprod is used. See Contribution I/2.                         */

 DIRBAK:PROC(K1,L,R,A,INT,Z) OPTIONS (REORDER);       /* PAGE 347 */    /*00000010*/
    DCL (K,K1,R,I,J,L, M,LBR,UBR) BIN FIXED(31),                        /*00000020*/
        (Z(*,*),                                                        /*00000030*/
         A(*,*)) BIN FLOAT(21),                                         /*00000040*/
         INT(*) BIN FIXED(15,0),                                        /*00000050*/
         X BIN FLOAT(21),                                               /*00000060*/
         SUMBOX BIN FLOAT(53);                                          /*00000070*/

/* Input to procedure dirbak
/*    k, l  parameters output by a procedure preparing A for reduction to
/*          Hessenberg form [3].
/*    r     the number of eigenvectors of the Hessenberg matrix which have
/*          been determined.
/*    a     the n x n array output by dirhes which includes the elements
/*          of N.
/*    int   an integer array describing the interchanges used in dirhes.
/*    z     an n x r array giving r eigenvectors of the derived Hessenberg
/*          matrix. Corresponding to a real eigenvalue there is one column
/*          of this array; corresponding to a complex conjugate pair of
/*          eigenvalues lambda+-imu the matrix z contains two consecutive
/*          columns a and b such that a+-ib are the vectors corresponding
/*          to lambda+-imu respectively.
/* Output from procedure dirbak
/*    z     an n x r array giving r eigenvectors of the matrix reduced by
/*          dirhes in an analogous form to that used in the input z. The
/*          output vectors are not normalized since usually they will be
/*          the eigenvectors of the prepared version of the original
/*          matrix and will need further back transformation to give
/*          eigenvectors of the original matrix (see [3]).                 */

    LBR=LBOUND(Z,2);  UBR=LBR+R-1;                                      /*00000090*/
    K=K1+2;                                                             /*00000100*/
    /*FORM N*Z */                                                       /*00000110*/
    DO J=LBR TO UBR;                                                    /*00000120*/
       DO I=L TO K BY -1;                                               /*00000130*/
          SUMBOX=Z(I,J);                                                /*00000140*/
          DO M=K-1 TO I-1;                                              /*00000150*/
             SUMBOX=SUMBOX+A(I,M-1)*Z(M,J);                             /*00000160*/
          END;                                                          /*00000170*/
          Z(I,J)=SUMBOX;                                                /*00000180*/
       END;                                                             /*00000190*/
    END;                                                                /*00000200*/
    K=K-1;                                                              /*00000210*/
    /* INTERCHANGE WHERE NECESSARY */                                   /*00000220*/
    DO I=L TO K BY -1;                                                  /*00000230*/
       M=INT(I);                                                        /*00000240*/
       IF M^=I THEN DO J=LBR TO UBR;                                    /*00000250*/
                       X=Z(M,J); Z(M,J)=Z(I,J); Z(I,J)=X;               /*00000260*/
                    END;                                                /*00000270*/
    END;                                                                /*00000280*/
 END DIRBAK;                                                            /*00000290*/

/* P. 347 */

/* Given the complex unsymmetric matrix, A, stored in the array
   a[1:n, 1:n], this procedure reduces the sub-matrix
   of order l-k+1, which starts at the element a[k,k]
   and finishes at the element a[l,l], to Hessenberg form,
   H, by non-orthogonal elementary transformations.  The matrix H is
   overwritten on A with details of the transformations stored in the
   remaining triangle under H and in the array int[k:l].  The procedure
   CDIV is used. */

 COMHES:PROC(N,K,L,A,INT) OPTIONS (REORDER);      /*PAGE 347*/          /*00000010*/
    DCL (N,K,L,I,J,LA,M,LBN,UBN) BIN FIXED (31,0),                      /*00000020*/
         INT(*) BIN FIXED (15,0),                                       /*00000030*/
        (X,Y) BIN FLOAT(21) COMPLEX,                                    /*00000040*/
         A(*,*) BIN FLOAT(21) COMPLEX;                                  /*00000050*/

/* Input to procedure comhes                                               */
/*    n     order of the full complex matrix A .                           */
/*    k,l   parameters output by a procedure for preparing A [3]. If A is  */
/*          not prepared in this way then k = 1, l = n.                    */
/*    A     An n x n array, containing real and imaginary parts of A.      */
/* Output from procedure comhes                                            */
/*    A     an n x n array containing the real and imaginary parts of the  */
/*          derived Hessenberg matrix.  The multipliers used in the        */
/*          reduction are also stored in these arrays as n(i,r+1).         */
/*    int   an integer array describing the row and column interchanges    */
/*          involved in the reduction.                                     */

    LBN=LBOUND(A,1);  UBN=LBN+N-1;                                      /*00000060*/
    LA=L-1;                                                             /*00000070*/
    DO M=K+1 TO LA;                                                     /*00000080*/
       I=M;X=0E0+0E0I;                                                  /*00000090*/
       DO J=M TO L;                                                     /*00000100*/
          IF ABS(REAL(A(J,M-1)))+ABS(IMAG(A(J,M-1)))>                   /*00000110*/
             ABS(REAL(X))+ABS(IMAG(X)) THEN DO;                         /*00000120*/
                                               X=A(J,M-1);              /*00000130*/
                                               I=J;                     /*00000140*/
                                            END;                        /*00000150*/
       END;                                                             /*00000160*/
       INT(M)=I;                                                        /*00000170*/
       IF I^=M THEN DO;                                                 /*00000180*/
                       /*INTERCHANGE ROWS AND COLS OF ARRAY A */        /*00000190*/
                       DO J=M-1 TO UBN;                                 /*00000200*/
                          Y=A(I,J); A(I,J)=A(M,J); A(M,J)=Y;            /*00000210*/
                       END;                                             /*00000220*/
                       DO J=LBN TO L;                                   /*00000230*/
                          Y=A(J,I); A(J,I)=A(J,M); A(J,M)=Y;            /*00000240*/
                       END;                                             /*00000250*/
                    END;/*END INTERCHANGE */                            /*00000260*/
       IF X^=0E0+0E0I THEN DO I=M+1 TO L;                               /*00000270*/
                              Y=A(I,M-1);                               /*00000280*/
                              IF Y^=0E0+0E0I THEN DO;                   /*00000290*/
                                 Y,A(I,M-1)=Y/X;                        /*00000300*/
                                 DO J=M TO UBN;                         /*00000310*/
                                    A(I,J)=A(I,J)-Y*A(M,J);             /*00000320*/
                                 END;                                   /*00000330*/
                                 DO J=LBN TO L;                         /*00000340*/
                                    A(J,M)=A(J,M)+Y*A(J,I);             /*00000350*/
                                 END;                                   /*00000360*/
                                                  END;                  /*00000370*/
                           END;                                         /*00000380*/
    END;                                                                /*00000390*/
 END COMHES;                                                            /*00000400*/

/* p. 348 */

/* Given r eigenvectors, stored as columns in the array z[1:n, 1:r],
   of the complex Hessenberg matrix, H, which was
   formed by COMHES, and details of the transformations as left by COMHES
   below H and in the array int[k:l]. 1 <= k <= l <= n, this procedure forms
   the eigenvectors of the matrix A and overwrites them on the given
   vectors. */

 COMBAK:PROC(K1,L,R,A,INT,Z) OPTIONS (REORDER);   /*PAGE 348*/          /*00000010*/
    DCL (K,L,R,I,J,M,K1,LBR,UBR) BIN FIXED(31),                         /*00000020*/
         INT(*) BIN FIXED(15),                                          /*00000030*/
         X BIN FLOAT(21) COMPLEX,                                       /*00000040*/
        (A,Z)(*,*) BIN FLOAT(21) COMPLEX;                               /*00000050*/

/* Input to procedure combak                                               */
/*    k, l  parameters output by the procedure used for preparing A [3].   */
/*    r     the number of eigenvectors of the derived complex Hessenberg   */
/*          matrix which have been determined.                             */
/*    A     the array output by comhes which include the real and          */
/*          imaginary parts of the row and column multipliers used in the  */
/*          reduction.                                                     */
/*    int   an integer array describing the row and column interchanges    */
/*          used in comhes.                                                */
/*    Z     an n x r array of which the r columns are the real parts and   */
/*          imaginary parts of r eigenvectors of the derived               */
/*          Hessenberg matrix. They are not normalized since usually they  */
/*          will belong to the prepared version of the matrix and will     */
/*          need further back transformation to give vectors of the        */
/*          original matrix (see [3]).                                     */
/* Output from procedure combak                                            */
/*    Z     An n x r array giving the real and imaginary parts of the      */
/*          eigenvectors of the matrix reduced by comhes.                  */

    LBR=LBOUND(A,2);  UBR=LBR+R-1;                                      /*00000060*/
    K=K1+1;                                                             /*00000070*/
    DO M=L-1 TO K BY -1;                                                /*00000080*/
       DO I=M+1 TO L;                                                   /*00000090*/
          X=A(I,M-1);                                                   /*00000100*/
          IF X^=0E0+0E0I THEN DO J=LBR TO UBR;                          /*00000110*/
                                 Z(I,J)=Z(I,J)+X*Z(M,J);                /*00000120*/
                              END;                                      /*00000130*/
       END;                                                             /*00000140*/
       I=INT(M);                                                        /*00000150*/
       IF I^=M THEN DO J=LBR TO UBR;                                    /*00000160*/
                       X=Z(I,J); Z(I,J)=Z(M,J); Z(M,J)=X;               /*00000170*/
                    END;                                                /*00000180*/
    END;                                                                /*00000190*/
 END COMBAK;                                                            /*00000200*/

/* P. 349 */

/* Given the unsymmetric matrix, A, stored in the array a[1:n, 1:n],       */
/* this procedure reduces the sub-matrix of order l-k+i, which starts      */
/* at the element a[k, k] and finishes at the element a[l, l], to          */
/* Hessenberg form, H, by orthogonal transformations. The matrix H is      */
/* overwritten on A with details of the transformations stored in the      */
/* remaining triangle under H and in the array d[k:l].  tol is a tolerance */
/* for checking if the transformation is valid.                            */

/* Tested with data on page 353, giving the results on page 354. 11/4/2008 */

 ORTHES:PROC(N,K,L,TOL,A,D) OPTIONS (REORDER);      /*PAGE 349*/        /*00000010*/
    DCL (N,K,L,I,J,M,LA,LBN,UBN) BIN FIXED (31),                        /*00000020*/
        (F,H) FLOAT(18),                                                /*00000030*/
        (G,TOL) FLOAT(18),                                              /*00000040*/
        (A(*,*),                                                        /*00000050*/
         D(*)) FLOAT(18);                                               /*00000060*/

/* Input to procedure orthes
/*    n     order of the full real matrix A.
/*    k, l  parameters output by a procedure preparing A for reduction to
/*          Hessenberg form [3]. If A has not been prepared then k = 1,
/*          l = n.
/*    tol   a machine dependent constant. Should be set equal to           */
/*          eta/macheps where eta is the smallest number representable in
/*          the computer and macheps is the smallest positive number for
/*          which 1 + macheps ^= 1.
/*    a     the n x n real matrix A normally in prepared form.
/* Output from procedure orthes
/*    a     the upper part of this n x n array gives the elements of the
/*          derived upper-Hessenberg matrix. The remaining elements give
/*          information on the orthogonal transformations for use in
/*          ortbak.
/*    d     a linear array giving the remaining information on the
/*          orthogonal transformations to be used in ortbak.               */

    LBN=LBOUND(A,1);  UBN=LBN+N-1;                                      /*00000070*/
    LA=L-1;                                                             /*00000080*/
    DO M=K+1 TO LA;                                                     /*00000090*/
       H=0E0;                                                           /*00000100*/
       DO I=L TO M BY -1;                                               /*00000110*/
          F,D(I)=A(I,M-1); H=H+F**2;                                    /*00000120*/
       END;                                                             /*00000130*/
       IF H<=TOL THEN DO;                                               /*00000140*/
                         G=0E0; GO TO SKIP;                             /*00000150*/
                      END;                                              /*00000160*/
       G=SQRT(H);                                                       /*00000170*/
       IF F>=0E0 THEN G=-G;                                             /*00000180*/
       H=H-F*G ; D(M)=F-G;                                              /*00000190*/
       /*FORM (I-(U*UT)/H)*A */                                         /*00000200*/
       DO J=M TO UBN;                                                   /*00000210*/
          F=0E0;                                                        /*00000220*/
          DO I=L TO M BY -1;                                            /*00000230*/
             F=F+D(I)*A(I,J);                                           /*00000240*/
          END;                                                          /*00000250*/
          F=F/H;                                                        /*00000260*/
          DO I=M TO L;                                                  /*00000270*/
             A(I,J)=A(I,J)-F*D(I);                                      /*00000280*/
          END;                                                          /*00000290*/
       END;                                                             /*00000300*/
       /*FORM(I-(U*UT)/H)*A*(I-(U*UT)/H) */                             /*00000310*/
       DO I=LBN TO L;                                                   /*00000320*/
          F=0E0;                                                        /*00000330*/
          DO J=L TO M BY -1;                                            /*00000340*/
             F=F+D(J)*A(I,J);                                           /*00000350*/
          END;                                                          /*00000360*/
          F=F/H;                                                        /*00000370*/
          DO J=M TO L;                                                  /*00000380*/
             A(I,J)=A(I,J)-F*D(J);                                      /*00000390*/
          END;                                                          /*00000400*/
       END;                                                             /*00000410*/
 SKIP:    A(M,M-1)=G;                                                   /*00000420*/
    END;                                                                /*00000430*/
 END ORTHES;                                                            /*00000440*/

/* Given r eigenvectors, stored as columns in the array z[1:n, 1:r], of    */
/* the Hessenberg matrix, H, which was formed by orthes, and details of    */
/* the transformations as left by orthes below H and in the array d[k:l],  */
/* 1 <= k <= l <= n, this procedure forms the eigenvectors of the matrix A */
/* and overwrites them on the given vectors. The array d is used as a      */
/* temporary store and is not restored to its original form.               */

 ORTBAK:PROC(K,L,R,A,D,Z) OPTIONS (REORDER);      /*PAGE 350*/          /*00000010*/
    DCL (K,L,R,I,J,M,MA,LBR,UBR) BIN FIXED(31),                         /*00000020*/
        (G,H) BIN FLOAT(53),                                            /*00000030*/
        (Z(*,*),                                                        /*00000040*/
         D(*),                                                          /*00000050*/
         A(*,*)) BIN FLOAT(21);                                         /*00000060*/

/* Input to procedure ortbak                                               */
/*    k, l  parameters output by the procedure used for preparing A [3].   */
/*    r     number of eigenvectors of the derived Hessenberg matrix which  */
/*          have been found.
/*    a     the n x n array output by orthes which includes information on */
/*          the orthogonal transformations.                                */
/*    d     a linear array output by orthes giving the remaining inform-   */
/*          ation on the orthogonal transformations.                       */
/*    z     an n x r array giving r eigenvectors of the derived Hessenberg */
/*          matrix.  Corresponding to a real eigenvalue there is one       */
/*          column of this array ; corresponding to a complex conjugate    */
/*          pair of eigenvalues lambda+-imu the matrix z contains two      */
/*          consecutive columns a and b such that a+-ib are the vectors    */
/*          corresponding to lambda+-imu respectively.                     */
/* Output from procedure ortbak                                            */
/*    z     an n x r array giving r eigenvectors of the matrix reduced     */
/*          by orthes in a form analogous to that used in the input z. The */
/*          output vectors are not normalized since usually they will      */
/*          belong to the prepared version of the original matrix and will */
/*          need further back transformation to give vectors of the        */
/*          original matrix. (See [3].)                                    */
/*    d     the original information in d is destroyed.                    */

    LBR=LBOUND(Z,2);  UBR=LBR+R-1;                                      /*00000070*/
    DO M=L-2 TO K BY -1;                                                /*00000080*/
       MA=M+1; H=A(MA,M);                                               /*00000090*/
       IF H=0E0 THEN GO TO SKIP;                                        /*00000100*/
       H=H*D(MA);                                                       /*00000110*/
       DO I=M+2 TO L;                                                   /*00000120*/
          D(I)=A(I,M);                                                  /*00000130*/
       END;                                                             /*00000140*/
       DO J=LBR TO UBR;                                                 /*00000150*/
          G=0E0;                                                        /*00000160*/
          DO I=MA TO L;                                                 /*00000170*/
             G=G+D(I)*Z(I,J);                                           /*00000180*/
          END;                                                          /*00000190*/
          G=G/H;                                                        /*00000200*/
          DO I=MA TO L;                                                 /*00000210*/
             Z(I,J)=Z(I,J)+G*D(I);                                      /*00000220*/
          END;                                                          /*00000230*/
       END;                                                             /*00000240*/
 SKIP:                                                                  /*00000250*/
    END;                                                                /*00000260*/
 END ORTBAK;                                                            /*00000270*/
