/************************************************************************/
/*                                                                      */
/* ALGORITHM 644: A Portable Package for Bessel Functions of a          */
/*                Complex Argument and Nonnegative Order.               */
/*                                                                      */
/* D. E. Amos, Numerical Mathematics Division,                          */
/* Sandia National Laboratories, Albuquerque, NM, 87185, U.S.A.         */
/*                                                                      */
/* ACM Transctions on Mathematical Software, Vol. 12 No. 3, Sept. 1986, */
/* pp. 265-273.                                                         */
/*                                                                      */
/************************************************************************/

/* This algorithm computes the Bessel functions H1(z), H2(z), I(z),      */
/* J(z), K(z), Y(z), and Airy functions Ai(z), Ai'(z), Bi(z), Bi'(z),    */
/* for orders v >= 0 and complex z in -pi < arg z <= pi.                 */

/*      REMARK ON ALGORITHM 644, COLLECTED ALGORITHMS FROM ACM.          */
/*      THIS WORK PUBLISHED IN TRANSACTIONS ON MATHEMATICAL SOFTWARE,    */
/*      VOL. 21, NO. 4, December, 1995, P.  388--393.                    */

/* Converted from Fortran 90 to PL/I by R. A. Vowels, 14 May 2007.       */

/* Error in Fortran version corrected in PL/I version 14 May 2007.       */
/* In procedure CBIRY, CONE was declared as REAL, but initialized to a   */
/* COMPLEX constant.  Context was that it must be COMPLEX.               */

/* NOTE: Initializeation of CIP is not consistent in three procedures.   */
/*       Needs checking. */

/* Code converted using TO_F90 by Alan Miller                            */
/* Date: 2002-02-08  Time: 17:53:05                                      */
/* Latest revision - 16 April 2002                                       */





(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbesh: PROCEDURE (z, fnu, kode, m, n, cy, nz, ierr) OPTIONS (REORDER);          
/* ***BEGIN PROLOGUE  CBESH                                              */
/* ***DATE WRITTEN   830501   (YYMMDD)                                   */
/* ***REVISION DATE  890801, 930101   (YYMMDD)                           */
/* ***CATEGORY NO.  B5K                                                  */
/* ***KEYWORDS  H-BESSEL FUNCTIONS,BESSEL FUNCTIONS OF COMPLEX ARGUMENT, */
/*             BESSEL FUNCTIONS OF THIRD KIND,HANKEL FUNCTIONS           */
/* ***AUTHOR  AMOS, DONALD E., SANDIA NATIONAL LABORATORIES              */
/* ***PURPOSE  TO COMPUTE THE H-BESSEL FUNCTIONS OF A COMPLEX ARGUMENT   */
/* ***DESCRIPTION                                                        */

/*   ON KODE=1, CBESH COMPUTES AN N MEMBER SEQUENCE OF COMPLEX           */
/*   HANKEL (BESSEL) FUNCTIONS CY(J)=H(M,FNU+J-1,Z) FOR KINDS M=1        */
/*   OR 2, REAL, NONNEGATIVE ORDERS FNU+J-1, J=1,...,N, AND COMPLEX      */
/*   Z.NE.CMPLX(0.0E0,0.0E0) IN THE CUT PLANE -PI < ARG(Z) <= PI.        */
/*   ON KODE=2, CBESH COMPUTES THE SCALED HANKEL FUNCTIONS               */

/*   CY(I)=H(M,FNU+J-1,Z)*EXP(-MM*Z*I)       MM=3-2M,      I**2=-1.      */

/*   WHICH REMOVES THE EXPONENTIAL BEHAVIOR IN BOTH THE UPPER            */
/*   AND LOWER HALF PLANES.  DEFINITIONS AND NOTATION ARE FOUND IN       */
/*   THE NBS HANDBOOK OF MATHEMATICAL FUNCTIONS (REF. 1).                */

/*   INPUT                                                               */
/*     Z      - Z=CMPLX(X,Y), Z.NE.CMPLX(0.,0.),-PI < ARG(Z) <= PI       */
/*     FNU    - ORDER OF INITIAL H FUNCTION, FNU >= 0.0E0                */
/*     KODE   - A PARAMETER TO INDICATE THE SCALING OPTION               */
/*              KODE= 1  RETURNS                                         */
/*                       CY(J)=H(M,FNU+J-1,Z),      J=1,...,N            */
/*                  = 2  RETURNS                                         */
/*                       CY(J)=H(M,FNU+J-1,Z)*EXP(-I*Z*(3-2M))           */
/*                            J=1,...,N  ,  I**2=-1                      */
/*     M      - KIND OF HANKEL FUNCTION, M=1 OR 2                        */
/*     N      - NUMBER OF MEMBERS OF THE SEQUENCE, N >= 1                */

/*   OUTPUT                                                              */
/*     CY     - A COMPLEX VECTOR WHOSE FIRST N COMPONENTS CONTAIN        */
/*              VALUES FOR THE SEQUENCE                                  */
/*              CY(J)=H(M,FNU+J-1,Z)  OR                                 */
/*              CY(J)=H(M,FNU+J-1,Z)*EXP(-I*Z*(3-2M))  J=1,...,N         */
/*              DEPENDING ON KODE, I**2=-1.                              */
/*     NZ     - NUMBER OF COMPONENTS SET TO ZERO DUE TO UNDERFLOW,       */
/*              NZ= 0   , NORMAL RETURN                                  */
/*              NZ > 0 , FIRST NZ COMPONENTS OF CY SET TO ZERO DUE TO UNDERFLOW, */
/*                        CY(J)=CMPLX(0.0,0.0) J=1,...,NZ WHEN Y > 0.0 AND M=1 */
/*                        OR Y < 0.0 AND M=2. FOR THE COMPLEMENTARY HALF PLANES, */
/*                        NZ STATES ONLY THE NUMBER OF UNDERFLOWS.       */
/*     IERR    -ERROR FLAG                                               */
/*              IERR=0, NORMAL RETURN - COMPUTATION COMPLETED            */
/*              IERR=1, INPUT ERROR   - NO COMPUTATION                   */
/*              IERR=2, OVERFLOW      - NO COMPUTATION, FNU+N-1 TOO      */
/*                      LARGE OR ABS(Z) TOO SMALL OR BOTH                */
/*              IERR=3, ABS(Z) OR FNU+N-1 LARGE - COMPUTATION DONE       */
/*                      BUT LOSSES OF SIGNIFICANCE BY ARGUMENT REDUCTION */
/*                      PRODUCE LESS THAN HALF OF MACHINE ACCURACY       */
/*              IERR=4, ABS(Z) OR FNU+N-1 TOO LARGE - NO COMPUTATION BECAUSE OF */
/*                      COMPLETE LOSSES OF SIGNIFICANCE BY ARGUMENT REDUCTION */
/*              IERR=5, ERROR              - NO COMPUTATION,             */
/*                      ALGORITHM TERMINATION CONDITION NOT MET          */

/* ***LONG DESCRIPTION                                                   */

/*    THE COMPUTATION IS CARRIED OUT BY THE RELATION                     */

/*    H(M,FNU,Z)=(1/MP)*EXP(-MP*FNU)*K(FNU,Z*EXP(-MP))                   */
/*        MP=MM*HPI*I,  MM=3-2*M,  HPI=PI/2,  I**2=-1                    */

/*    FOR M=1 OR 2 WHERE THE K BESSEL FUNCTION IS COMPUTED FOR THE       */
/*    RIGHT HALF PLANE RE(Z) >= 0.0. THE K FUNCTION IS CONTINUED         */
/*    TO THE LEFT HALF PLANE BY THE RELATION                             */

/*    K(FNU,Z*EXP(MP)) = EXP(-MP*FNU)*K(FNU,Z)-MP*I(FNU,Z)               */
/*    MP=MR*PI*I, MR=+1 OR -1, RE(Z) > 0, I**2=-1                        */

/*    WHERE I(FNU,Z) IS THE I BESSEL FUNCTION.                           */

/*    EXPONENTIAL DECAY OF H(M,FNU,Z) OCCURS IN THE UPPER HALF Z PLANE FOR */
/*    M=1 AND THE LOWER HALF Z PLANE FOR M=2.  EXPONENTIAL GROWTH OCCURS IN THE */
/*    COMPLEMENTARY HALF PLANES.  SCALING BY EXP(-MM*Z*I) REMOVES THE    */
/*    EXPONENTIAL BEHAVIOR IN THE WHOLE Z PLANE FOR Z TO INFINITY.       */

/*    FOR NEGATIVE ORDERS,THE FORMULAE                                   */

/*          H(1,-FNU,Z) = H(1,FNU,Z)*EXP( PI*FNU*I)                      */
/*          H(2,-FNU,Z) = H(2,FNU,Z)*EXP(-PI*FNU*I)                      */
/*                    I**2=-1                                            */

/*    CAN BE USED.                                                       */

/*    IN MOST COMPLEX VARIABLE COMPUTATION, ONE MUST EVALUATE ELEMENTARY */
/*    FUNCTIONS. WHEN THE MAGNITUDE OF Z OR FNU+N-1 IS LARGE, LOSSES OF  */
/*    SIGNIFICANCE BY ARGUMENT REDUCTION OCCUR.                          */
/*    CONSEQUENTLY, IF EITHER ONE EXCEEDS U1=SQRT(0.5/UR), THEN LOSSES   */
/*    EXCEEDING HALF PRECISION ARE LIKELY AND AN ERROR FLAG IERR=3 IS    */
/*    TRIGGERED WHERE UR=R1MACH(4)=UNIT ROUNDOFF.  ALSO IF EITHER IS LARGER */
/*    THAN U2=0.5/UR, THEN ALL SIGNIFICANCE IS LOST AND IERR=4.          */
/*    IN ORDER TO USE THE INT FUNCTION, ARGUMENTS MUST BE FURTHER RESTRICTED */
/*    NOT TO EXCEED THE LARGEST MACHINE INTEGER, U3=I1MACH(9).           */
/*    THUS, THE MAGNITUDE OF Z AND FNU+N-1 IS RESTRICTED BY MIN(U2,U3).  */
/*    ON 32 BIT MACHINES, U1,U2, AND U3 ARE APPROXIMATELY 2.0E+3, 4.2E+6, */
/*    2.1E+9 IN SINGLE PRECISION ARITHMETIC AND 1.3E+8, 1.8E+16, 2.1E+9 IN */
/*    DOUBLE PRECISION ARITHMETIC RESPECTIVELY.  THIS MAKES U2 AND U3    */
/*    LIMITING IN THEIR RESPECTIVE ARITHMETICS.  THIS MEANS THAT ONE CAN */
/*    EXPECT TO RETAIN, IN THE WORST CASES ON 32 BIT MACHINES, NO DIGITS */
/*    IN SINGLE AND ONLY 7 DIGITS IN DOUBLE PRECISION ARITHMETIC.        */
/*    SIMILAR CONSIDERATIONS HOLD FOR OTHER MACHINES.                    */

/*    THE APPROXIMATE RELATIVE ERROR IN THE MAGNITUDE OF A COMPLEX BESSEL */
/*    FUNCTION CAN BE EXPRESSED BY P*10**S WHERE P=MAX(UNIT ROUNDOFF,1.0E-18) */
/*    IS THE NOMINAL PRECISION AND 10**S REPRESENTS THE INCREASE IN ERROR DUE */
/*    TO ARGUMENT REDUCTION IN THE ELEMENTARY FUNCTIONS.                 */
/*    HERE, S=MAX(1, ABS(LOG10(ABS(Z))), ABS(LOG10(FNU))) APPROXIMATELY  */
/*    (I.E. S=MAX(1,ABS(EXPONENT OF ABS(Z),ABS(EXPONENT OF FNU)) ).      */
/*    HOWEVER, THE PHASE ANGLE MAY HAVE ONLY ABSOLUTE ACCURACY.          */
/*    THIS IS MOST LIKELY TO OCCUR WHEN ONE COMPONENT (IN ABSOLUTE VALUE) */
/*    IS LARGER THAN THE OTHER BY SEVERAL ORDERS OF MAGNITUDE.           */
/*    IF ONE COMPONENT IS 10**K LARGER THAN THE OTHER, THEN ONE CAN EXPECT */
/*    ONLY MAX(ABS(LOG10(P))-K, 0) SIGNIFICANT DIGITS; OR, STATED ANOTHER */
/*    WAY, WHEN K EXCEEDS THE EXPONENT OF P, NO SIGNIFICANT DIGITS REMAIN IN */
/*    THE SMALLER COMPONENT.  HOWEVER, THE PHASE ANGLE RETAINS ABSOLUTE  */
/*    ACCURACY BECAUSE, IN COMPLEX ARITHMETIC WITH PRECISION P, THE SMALLER */
/*    COMPONENT WILL NOT (AS A RULE) DECREASE BELOW P TIMES THE MAGNITUDE */
/*    OF THE LARGER COMPONENT.  IN THESE EXTREME CASES, THE PRINCIPAL PHASE */
/*    ANGLE IS ON THE ORDER OF +P, -P, PI/2-P, OR -PI/2+P.               */

/* ***REFERENCES  HANDBOOK OF MATHEMATICAL FUNCTIONS BY M. ABRAMOWITZ AND */
/*        I. A. STEGUN, NBS AMS SERIES 55, U.S. DEPT. OF COMMERCE, 1955. */

/*      COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT              */
/*        BY D. E. AMOS, SAND83-0083, MAY 1983.                          */

/*      COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT              */
/*        AND LARGE ORDER BY D. E. AMOS, SAND83-0643, MAY 1983           */

/*      A SUBROUTINE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT  */
/*        AND NONNEGATIVE ORDER BY D. E. AMOS, SAND85-1018, MAY, 1985    */

/*      A PORTABLE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT    */
/*        AND NONNEGATIVE ORDER BY D. E. AMOS, ACM TRANS. MATH. SOFTWARE, */
/*        VOL. 12, NO. 3, SEPTEMBER 1986, PP 265-273.                    */

/* ***ROUTINES CALLED  CACON,CBKNU,CBUNK,CUOIK,I1MACH,R1MACH             */
/* ***END PROLOGUE  CBESH                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( m )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( cy(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( ierr )  FIXED BINARY (31);

   DECLARE ( zn, zt, csgn )  COMPLEX FLOAT (18);
   DECLARE ( aa, alim, aln, arg, az, cpn, dig, elim, fmm, fn, fnul, rhpi, 
      rl, r1m5, sgn, spn, tol, ufl, xn, xx, yn, yy, bb, ascle, rtol, atol )  
      FLOAT (18);
   DECLARE ( i, inu, inuh, ir, k, k1, k2, mm, mr, nn, nuf, nw )  FIXED BINARY (31);

   DECLARE ( hpi  STATIC INITIAL ( 1.57079632679489662E0) )  FLOAT (18);
   DECLARE Debug BIT(1) STATIC INITIAL ('0'B);

/* ***FIRST EXECUTABLE STATEMENT  CBESH                                  */
   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, inu, inuh );
      PUT SKIP DATA ( ir, k, k1 );
      PUT SKIP DATA ( k2, mm, mr );
      PUT SKIP DATA ( nn, nuf, nw );
      RESIGNAL;
   END;

IF DEBUG THEN PUT SKIP LIST ('ENTERED CBESH');

   nz = 0;
   xx = REAL(z);
   yy = IMAG(z);
   ierr = 0;
   IF  xx  = 0   &   yy  = 0 THEN
      ierr = 1;
   IF  fnu < 0 THEN
      ierr = 1;
   IF  m < 1  |   m > 2 THEN
      ierr = 1;
   IF  kode < 1  |   kode > 2 THEN
      ierr = 1;
   IF  n < 1 THEN
      ierr = 1;
   IF  ierr ^= 0 THEN
      RETURN;
   nn = n;
/* ----------------------------------------------------------------------- */
/*     SET PARAMETERS RELATED TO MACHINE CONSTANTS.                      */
/*     TOL IS THE APPROXIMATE UNIT ROUNDOFF LIMITED TO 1.0E-18.          */
/*     ELIM IS THE APPROXIMATE EXPONENTIAL OVER- AND UNDERFLOW LIMIT.    */
/*     EXP(-ELIM) < EXP(-ALIM)=EXP(-ELIM)/TOL    AND                     */
/*     EXP(ELIM) > EXP(ALIM)=EXP(ELIM)*TOL       ARE INTERVALS NEAR      */
/*     UNDERFLOW AND OVERFLOW LIMITS WHERE SCALED ARITHMETIC IS DONE.    */
/*     RL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC EXPANSION FOR LARGE Z. */
/*     DIG = NUMBER OF BASE 10 DIGITS IN TOL = 10**(-DIG).               */
/*     FNUL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC SERIES FOR LARGE FNU */
/* ----------------------------------------------------------------------- */
   tol = MAX(EPSILON(0.00000000000000000E0), 1.00000000000000000E-18);
   k1 = MINEXPONENT(0.00000000000000000E0);
   k2 = MAXEXPONENT(0.00000000000000000E0);
   r1m5 = LOG10( RADIX(0.00000000000000000E0) );
   k = MIN(ABS(k1), ABS(k2));
   elim = 2.30300000000000000E0 * (k*r1m5 - 3.00000000000000000E0);
   k1 = DIGITS(0.00000000000000000E0) - 1;
   aa = r1m5 * k1;
   dig = MIN(aa, 18.0000000000000000E0);
   aa = aa * 2.30300000000000000E0;
   alim = elim + MAX(-aa, -41.4500000000000000E0);
   fnul = 10.0000000000000000E0 + 6.00000000000000000E0 * (dig - 3.00000000000000000E0); 

IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT A');

   rl = 1.20000000000000000E0 * dig + 3.00000000000000000E0;
   fn = fnu + (nn-1);
   mm = 3 - m - m;
   fmm = mm;
   zn = z * COMPLEX(0, -fmm);
   xn = REAL(zn);
   yn = IMAG(zn);
   az = ABS(z);
IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT B');

/* ----------------------------------------------------------------------- */
/*     TEST FOR RANGE                                                    */
/* ----------------------------------------------------------------------- */
   aa = 0.50000000000000000E0 / tol;
   bb = HUGE(0) * 0.50000000000000000E0;
   aa = MIN(aa,bb);
   IF  az <= aa  THEN
      DO;
         IF  fn <= aa  THEN
            DO;
               aa = SQRT(aa);
               IF  az > aa THEN
                  ierr = 3;
               IF  fn > aa THEN
                  ierr = 3;
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST ON THE LAST MEMBER OF THE SEQUENCE                  */
/* ----------------------------------------------------------------------- */
               ufl = TINY(0.00000000000000000E0) * 1.00000000000000000E+3;
IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT B1');
               IF  az >= ufl  THEN
                  DO;
                     IF  fnu <= fnul  THEN
                        DO;
                           IF  fn > 1.00000000000000000E0  THEN
                              DO;
                                 IF  fn <= 2.00000000000000000E0  THEN
                                    DO;
                                       IF  az > tol THEN  GO TO L10;
                                       arg = 0.50000000000000000E0 * az;
                                       aln = -fn * LOG(arg);
                                       IF  aln > elim THEN  GO TO L50;
                                    END;
                                 ELSE
                                    DO;
                                       CALL cuoik(zn, fnu, kode, 2, nn, cy, nuf, tol, elim, alim);
IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT B2');
                                       IF  nuf < 0 THEN  GO TO L50;
                                       nz = nz + nuf;
                                       nn = nn - nuf;
/* ----------------------------------------------------------------------- */
/*     HERE NN=N OR NN=0 SINCE NUF=0,NN, OR -1 ON RETURN FROM CUOIK      */
/*     IF NUF=NN, THEN CY(I)=CZERO FOR ALL I                             */
/* ----------------------------------------------------------------------- */
IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT B3');
                                       IF  nn  = 0 THEN  GO TO L40;
                                    END;
                              END;

L10:
                           IF  ^ (xn < 0  |   (xn  = 0   &  yn < 0 & m = 2))  THEN
                              DO;
/* ----------------------------------------------------------------------- */
/*     RIGHT HALF PLANE COMPUTATION, XN >= 0.  .AND.  (XN.NE.0.  .OR.    */
/*     YN >= 0.  .OR.  M=1)                                              */
/* ----------------------------------------------------------------------- */
                                 CALL cbknu(zn, fnu, kode, nn, cy, nz, tol, elim, alim);
                                 GO TO L20;
                              END;
/* ----------------------------------------------------------------------- */
/*     LEFT HALF PLANE COMPUTATION                                       */
/* ----------------------------------------------------------------------- */
                           mr = -mm;
                           CALL cacon(zn, fnu, kode, mr, nn, cy, nw, rl, fnul, tol, elim, alim);
                           IF  nw < 0 THEN  GO TO L60;
                           nz = nw;
                        END;
                     ELSE
                        DO;
/* ----------------------------------------------------------------------- */
/*     UNIFORM ASYMPTOTIC EXPANSIONS FOR FNU > FNUL                      */
/* ----------------------------------------------------------------------- */
                           mr = 0;
                           IF  ^ (xn >= 0   &   (xn ^= 0  
                              |   yn >= 0  |  m ^= 2))  THEN
                              DO;
                                 mr = -mm;
                                 IF  xn  = 0   &   yn < 0 THEN
                                    zn = -zn;
                              END;
                           CALL cbunk(zn, fnu, kode, mr, nn, cy, nw, tol, elim, alim);
IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT D');
                           IF  nw < 0 THEN  GO TO L60;
                           nz = nz + nw;
                        END;
/* ----------------------------------------------------------------------- */
/*     H(M,FNU,Z) = -FMM*(I/HPI)*(ZT**FNU)*K(FNU,-Z*ZT)                  */

/*     ZT=EXP(-FMM*HPI*I) = CMPLX(0.0,-FMM), FMM=3-2*M, M=1,2            */
/* ----------------------------------------------------------------------- */
L20:
                     sgn = SIGN(hpi,-fmm);
/* ----------------------------------------------------------------------- */
/*     CALCULATE EXP(FNU*HPI*I) TO MINIMIZE LOSSES OF SIGNIFICANCE       */
/*     WHEN FNU IS LARGE                                                 */
/* ----------------------------------------------------------------------- */
IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT E');
                     inu = fnu;
                     inuh = inu / 2;
IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT E1');
                     ir = inu - 2 * inuh;
                     arg = (fnu - (inu-ir)) * sgn;
                     rhpi = 1.00000000000000000E0 / sgn;
                     cpn = rhpi * COS(arg);
                     spn = rhpi * SIN(arg);
/*     ZN = CMPLX(-SPN,CPN)                                              */
                     csgn = COMPLEX(-spn, cpn );
/*     IF (MOD(INUH,2).EQ.1) ZN = -ZN                                    */
                     IF  REM(inuh,2)  = 1 THEN
                        csgn = -csgn;
                     zt = COMPLEX(0, -fmm);
                     rtol = 1.00000000000000000E0 / tol;
                     ascle = ufl * rtol;
IF DEBUG THEN PUT SKIP LIST ('CBESH: CHECKPOINT F');
                     DO  i = 1 TO  nn;
/*       CY(I) = CY(I)*ZN                                                */
/*       ZN = ZN*ZT                                                      */
                        zn = cy(i);
                        aa = REAL(zn);
                        bb = IMAG(zn);
                        atol = 1;
                        IF  MAX(ABS(aa),ABS(bb)) <= ascle  THEN
                           DO;
                              zn = zn * rtol;
                              atol = tol;
                           END;
                        zn = zn * csgn;
                        cy(i) = zn * atol;
                        csgn = csgn * zt;
                     END;
                     RETURN;

L40:
                     IF  xn >= 0 THEN
                        RETURN;
                  END;

L50:
               ierr = 2;
               nz = 0;
               RETURN;

L60:
               IF  nw  = -1 THEN  GO TO L50;
               nz = 0;
               ierr = 5;
               RETURN;
            END;
      END;
   nz = 0;
   ierr = 4;
   RETURN;

DECLARE SIGN GENERIC (DSIGN WHEN (FLOAT, FLOAT),
   ISIGN WHEN (FIXED, FIXED) );
ISIGN: PROCEDURE (X, Y) RETURNS (FIXED BINARY(31)) OPTIONS (INLINE);
   DECLARE (X, Y) FIXED BINARY (31);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END ISIGN;
DSIGN: PROCEDURE (X, Y) RETURNS (FLOAT (18)) OPTIONS (INLINE);
   DECLARE (X, Y) FLOAT (18);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END DSIGN;

   END cbesh;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbesi: PROCEDURE (z, fnu, kode, n, cy, nz, ierr) OPTIONS (REORDER);             
/* ***BEGIN PROLOGUE  CBESI                                              */
/* ***DATE WRITTEN   830501   (YYMMDD)                                   */
/* ***REVISION DATE  890801, 930101   (YYMMDD)                           */
/* ***CATEGORY NO.  B5K                                                  */
/* ***KEYWORDS  I-BESSEL FUNCTION,COMPLEX BESSEL FUNCTION,               */
/*             MODIFIED BESSEL FUNCTION OF THE FIRST KIND                */
/* ***AUTHOR  AMOS, DONALD E., SANDIA NATIONAL LABORATORIES              */
/* ***PURPOSE  TO COMPUTE I-BESSEL FUNCTIONS OF COMPLEX ARGUMENT         */
/* ***DESCRIPTION                                                        */

/*   ON KODE=1, CBESI COMPUTES AN N MEMBER SEQUENCE OF COMPLEX           */
/*   BESSEL FUNCTIONS CY(J)=I(FNU+J-1,Z) FOR REAL (dp), NONNEGATIVE      */
/*   ORDERS FNU+J-1, J=1,...,N AND COMPLEX Z IN THE CUT PLANE            */
/*   -PI < ARG(Z) <= PI. ON KODE=2, CBESI RETURNS THE SCALED FUNCTIONS   */

/*   CY(J)=EXP(-ABS(X))*I(FNU+J-1,Z)   J = 1,...,N , X=REAL(Z)           */

/*   WITH THE EXPONENTIAL GROWTH REMOVED IN BOTH THE LEFT AND            */
/*   RIGHT HALF PLANES FOR Z TO INFINITY. DEFINITIONS AND                */
/*   NOTATION ARE FOUND IN THE NBS HANDBOOK OF MATHEMATICAL              */
/*   FUNCTIONS (REF.1)                                                   */

/*   INPUT                                                               */
/*     Z      - Z=CMPLX(X,Y),  -PI < ARG(Z) <= PI                        */
/*     FNU    - ORDER OF INITIAL I FUNCTION, FNU >= 0.0                  */
/*     KODE   - A PARAMETER TO INDICATE THE SCALING OPTION               */
/*              KODE= 1  RETURNS                                         */
/*                       CY(J)=I(FNU+J-1,Z), J=1,...,N                   */
/*                  = 2  RETURNS                                         */
/*                       CY(J)=I(FNU+J-1,Z)*EXP(-ABS(X)), J=1,...,N      */
/*     N      - NUMBER OF MEMBERS OF THE SEQUENCE, N >= 1                */

/*   OUTPUT                                                              */
/*     CY     - A COMPLEX VECTOR WHOSE FIRST N COMPONENTS CONTAIN        */
/*              VALUES FOR THE SEQUENCE                                  */
/*              CY(J)=I(FNU+J-1,Z)  OR                                   */
/*              CY(J)=I(FNU+J-1,Z)*EXP(-ABS(X))  J=1,...,N               */
/*              DEPENDING ON KODE, X=REAL(Z)                             */
/*     NZ     - NUMBER OF COMPONENTS SET TO ZERO DUE TO UNDERFLOW,       */
/*              NZ= 0   , NORMAL RETURN                                  */
/*              NZ > 0 , LAST NZ COMPONENTS OF CY SET TO ZERO            */
/*                        DUE TO UNDERFLOW, CY(J)=CMPLX(0.0,0.0),        */
/*                        J = N-NZ+1,...,N                               */
/*     IERR   - ERROR FLAG                                               */
/*              IERR=0, NORMAL RETURN - COMPUTATION COMPLETED            */
/*              IERR=1, INPUT ERROR   - NO COMPUTATION                   */
/*              IERR=2, OVERFLOW      - NO COMPUTATION, REAL(Z) TOO      */
/*                      LARGE ON KODE=1                                  */
/*              IERR=3, ABS(Z) OR FNU+N-1 LARGE - COMPUTATION DONE       */
/*                      BUT LOSSES OF SIGNIFICANCE BY ARGUMENT           */
/*                      REDUCTION PRODUCE LESS THAN HALF OF MACHINE      */
/*                      ACCURACY                                         */
/*              IERR=4, ABS(Z) OR FNU+N-1 TOO LARGE - NO COMPUTA-        */
/*                      TION BECAUSE OF COMPLETE LOSSES OF SIGNIFI-      */
/*                      CANCE BY ARGUMENT REDUCTION                      */
/*              IERR=5, ERROR              - NO COMPUTATION,             */
/*                      ALGORITHM TERMINATION CONDITION NOT MET          */

/* ***LONG DESCRIPTION                                                   */

/*   THE COMPUTATION IS CARRIED OUT BY THE POWER SERIES FOR              */
/*   SMALL ABS(Z), THE ASYMPTOTIC EXPANSION FOR LARGE ABS(Z),            */
/*   THE MILLER ALGORITHM NORMALIZED BY THE WRONSKIAN AND A              */
/*   NEUMANN SERIES FOR IMTERMEDIATE MAGNITUDES, AND THE                 */
/*   UNIFORM ASYMPTOTIC EXPANSIONS FOR I(FNU,Z) AND J(FNU,Z)             */
/*   FOR LARGE ORDERS. BACKWARD RECURRENCE IS USED TO GENERATE           */
/*   SEQUENCES OR REDUCE ORDERS WHEN NECESSARY.                          */

/*   THE CALCULATIONS ABOVE ARE DONE IN THE RIGHT HALF PLANE AND         */
/*   CONTINUED INTO THE LEFT HALF PLANE BY THE FORMULA                   */

/*   I(FNU,Z*EXP(M*PI)) = EXP(M*PI*FNU)*I(FNU,Z)  REAL(Z) > 0.0          */
/*                 M = +I OR -I,  I**2=-1                                */

/*   FOR NEGATIVE ORDERS,THE FORMULA                                     */

/*        I(-FNU,Z) = I(FNU,Z) + (2/PI)*SIN(PI*FNU)*K(FNU,Z)             */

/*   CAN BE USED.  HOWEVER,FOR LARGE ORDERS CLOSE TO INTEGERS, THE FUNCTION */
/*   CHANGES RADICALLY.  WHEN FNU IS A LARGE POSITIVE INTEGER,THE MAGNITUDE OF */
/*   I(-FNU,Z) = I(FNU,Z) IS A LARGE NEGATIVE POWER OF TEN.  BUT WHEN FNU IS NOT */
/*   AN INTEGER, K(FNU,Z) DOMINATES IN MAGNITUDE WITH A LARGE POSITIVE POWER OF */
/*   TEN AND THE MOST THAT THE SECOND TERM CAN BE REDUCED IS BY UNIT ROUNDOFF */
/*   FROM THE COEFFICIENT. THUS, WIDE CHANGES CAN OCCUR WITHIN UNIT ROUNDOFF OF */
/*   A LARGE INTEGER FOR FNU.  HERE, LARGE MEANS FNU > ABS(Z).           */

/*   IN MOST COMPLEX VARIABLE COMPUTATION, ONE MUST EVALUATE ELEMENTARY  */
/*   FUNCTIONS.  WHEN THE MAGNITUDE OF Z OR FNU+N-1 IS                   */
/*   LARGE, LOSSES OF SIGNIFICANCE BY ARGUMENT REDUCTION OCCUR.          */
/*   CONSEQUENTLY, IF EITHER ONE EXCEEDS U1=SQRT(0.5/UR), THEN           */
/*   LOSSES EXCEEDING HALF PRECISION ARE LIKELY AND AN ERROR FLAG        */
/*   IERR=3 IS TRIGGERED WHERE UR=EPSILON(0.0)=UNIT ROUNDOFF.  ALSO      */
/*   IF EITHER IS LARGER THAN U2=0.5/UR, THEN ALL SIGNIFICANCE IS        */
/*   LOST AND IERR=4. IN ORDER TO USE THE INT FUNCTION, ARGUMENTS        */
/*   MUST BE FURTHER RESTRICTED NOT TO EXCEED THE LARGEST MACHINE        */
/*   INTEGER, U3=HUGE(0). THUS, THE MAGNITUDE OF Z AND FNU+N-1 IS        */
/*   RESTRICTED BY MIN(U2,U3). ON 32 BIT MACHINES, U1,U2, AND U3         */
/*   ARE APPROXIMATELY 2.0E+3, 4.2E+6, 2.1E+9 IN SINGLE PRECISION        */
/*   ARITHMETIC AND 1.3E+8, 1.8E+16, 2.1E+9 IN DOUBLE PRECISION          */
/*   ARITHMETIC RESPECTIVELY. THIS MAKES U2 AND U3 LIMITING IN           */
/*   THEIR RESPECTIVE ARITHMETICS. THIS MEANS THAT ONE CAN EXPECT        */
/*   TO RETAIN, IN THE WORST CASES ON 32 BIT MACHINES, NO DIGITS         */
/*   IN SINGLE AND ONLY 7 DIGITS IN DOUBLE PRECISION ARITHMETIC.         */
/*   SIMILAR CONSIDERATIONS HOLD FOR OTHER MACHINES.                     */

/*   THE APPROXIMATE RELATIVE ERROR IN THE MAGNITUDE OF A COMPLEX BESSEL */
/*   FUNCTION CAN BE EXPRESSED BY P*10**S WHERE P = MAX(UNIT ROUNDOFF,1.0E-18) */
/*   IS THE NOMINAL PRECISION AND 10**S REPRESENTS THE INCREASE IN ERROR DUE TO */
/*   ARGUMENT REDUCTION IN THE ELEMENTARY FUNCTIONS.  HERE, S =          */
/*   MAX(1,ABS(LOG10(ABS(Z))), ABS(LOG10(FNU))) APPROXIMATELY            */
/*   (I.E. S = MAX(1,ABS(EXPONENT OF ABS(Z), ABS(EXPONENT OF FNU)) ).    */
/*   HOWEVER, THE PHASE ANGLE MAY HAVE ONLY ABSOLUTE ACCURACY.           */
/*   THIS IS MOST LIKELY TO OCCUR WHEN ONE COMPONENT (IN ABSOLUTE VALUE) IS */
/*   LARGER THAN THE OTHER BY SEVERAL ORDERS OF MAGNITUDE.               */
/*   IF ONE COMPONENT IS 10**K LARGER THAN THE OTHER, THEN ONE CAN EXPECT ONLY */
/*   MAX(ABS(LOG10(P))-K, 0) SIGNIFICANT DIGITS; OR, STATED ANOTHER WAY, WHEN K */
/*   EXCEEDS THE EXPONENT OF P, NO SIGNIFICANT DIGITS REMAIN IN THE SMALLER */
/*   COMPONENT.  HOWEVER, THE PHASE ANGLE RETAINS ABSOLUTE ACCURACY BECAUSE, */
/*   IN COMPLEX ARITHMETIC WITH PRECISION P, THE SMALLER COMPONENT WILL NOT */
/*   (AS A RULE) DECREASE BELOW P TIMES THE MAGNITUDE OF THE LARGER COMPONENT. */
/*   IN THESE EXTREME CASES, THE PRINCIPAL PHASE ANGLE IS ON THE ORDER OF */
/*   +P, -P, PI/2-P, OR -PI/2+P.                                         */

/* ***REFERENCES  HANDBOOK OF MATHEMATICAL FUNCTIONS BY M. ABRAMOWITZ AND */
/*         I. A. STEGUN, NBS AMS SERIES 55, U.S. DEPT. OF COMMERCE, 1955. */

/*       COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT             */
/*         BY D. E. AMOS, SAND83-0083, MAY 1983.                         */

/*       COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT             */
/*         AND LARGE ORDER BY D. E. AMOS, SAND83-0643, MAY 1983          */

/*       A SUBROUTINE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT */
/*         AND NONNEGATIVE ORDER BY D. E. AMOS, SAND85-1018, MAY 1985    */

/*       A PORTABLE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT   */
/*         AND NONNEGATIVE ORDER BY D. E. AMOS, ACM TRANS. MATH. SOFTWARE, */
/*         VOL. 12, NO. 3, SEPTEMBER 1986, PP 265-273.                   */

/* ***ROUTINES CALLED  CBINU,I1MACH,R1MACH                               */
/* ***END PROLOGUE  CBESI                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( cy(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( ierr )  FIXED BINARY (31);

   DECLARE ( csgn, zn )  COMPLEX FLOAT (18);
   DECLARE ( aa, alim, arg, dig, elim, fnul, rl, r1m5, s1, s2, tol, xx, yy, 
      az, fn, bb, ascle, rtol, atol )  FLOAT (18);
   DECLARE ( i, inu, k, k1, k2, nn )  FIXED BINARY (31);

   DECLARE ( pi  STATIC INITIAL ( 3.14159265358979324E0) )  FLOAT (18);
   DECLARE cone  STATIC INITIAL ( 1.00000000000000000E0+0.00000000000000000E0i )
           COMPLEX FLOAT (18);

/* ***FIRST EXECUTABLE STATEMENT  CBESI                                  */
   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, inu, k );
      PUT SKIP DATA ( k1, k2, nn );
      RESIGNAL;
   END;

   PUT SKIP LIST ('ENTERED CBESI');

   ierr = 0;
   nz = 0;
   IF  fnu < 0 THEN
      ierr = 1;
   IF  kode < 1  |   kode > 2 THEN
      ierr = 1;
   IF  n < 1 THEN
      ierr = 1;
   IF  ierr ^= 0 THEN
      RETURN;
   xx = REAL(z);
   yy = IMAG(z);
/* ----------------------------------------------------------------------- */
/*     SET PARAMETERS RELATED TO MACHINE CONSTANTS.                      */
/*     TOL IS THE APPROXIMATE UNIT ROUNDOFF LIMITED TO 1.0E-18.          */
/*     ELIM IS THE APPROXIMATE EXPONENTIAL OVER- AND UNDERFLOW LIMIT.    */
/*     EXP(-ELIM) < EXP(-ALIM)=EXP(-ELIM)/TOL    AND                     */
/*     EXP(ELIM) > EXP(ALIM)=EXP(ELIM)*TOL       ARE INTERVALS NEAR      */
/*     UNDERFLOW AND OVERFLOW LIMITS WHERE SCALED ARITHMETIC IS DONE.    */
/*     RL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC EXPANSION FOR LARGE Z. */
/*     DIG = NUMBER OF BASE 10 DIGITS IN TOL = 10**(-DIG).               */
/*     FNUL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC SERIES FOR LARGE FNU. */
/* ----------------------------------------------------------------------- */
   tol = MAX(EPSILON(0.00000000000000000E0), 1.00000000000000000E-18);
   k1 = MINEXPONENT(0.00000000000000000E0);
   k2 = MAXEXPONENT(0.00000000000000000E0);
   r1m5 = LOG10( RADIX(0.00000000000000000E0) );
   k = MIN(ABS(k1),ABS(k2));
   elim = 2.30300000000000000E0 * (k*r1m5 - 3.00000000000000000E0);
   k1 = DIGITS(0.00000000000000000E0) - 1;
   aa = r1m5 * k1;
   dig = MIN(aa, 18.0000000000000000E0);
   aa = aa * 2.30300000000000000E0;
   alim = elim + MAX(-aa, -41.4500000000000000E0);
   rl = 1.20000000000000000E0 * dig + 3.00000000000000000E0;
   fnul = 10.0000000000000000E0 + 6.00000000000000000E0 * (dig - 3.00000000000000000E0); 
       
   az = ABS(z);
/* ----------------------------------------------------------------------- */
/*     TEST FOR RANGE                                                    */
/* ----------------------------------------------------------------------- */
   aa = 0.50000000000000000E0 / tol;
   bb = HUGE(0) * 0.50000000000000000E0;
   aa = MIN(aa,bb);
   IF  az <= aa  THEN
      DO;
         fn = fnu + (n-1);
         IF  fn <= aa  THEN
            DO;
               aa = SQRT(aa);
               IF  az > aa THEN
                  ierr = 3;
               IF  fn > aa THEN
                  ierr = 3;
               zn = z;
               csgn = cone;
               IF  xx < 0  THEN
                  DO;
                     zn = -z;
/* ----------------------------------------------------------------------- */
/*     CALCULATE CSGN=EXP(FNU*PI*I) TO MINIMIZE LOSSES OF SIGNIFICANCE   */
/*     WHEN FNU IS LARGE                                                 */
/* ----------------------------------------------------------------------- */
                     inu = fnu;
                     arg = (fnu - inu) * pi;
                     IF  yy < 0 THEN
                        arg = -arg;
                     s1 = COS(arg);
                     s2 = SIN(arg);
                     csgn = COMPLEX(s1, s2);
                     IF  REM(inu,2)  = 1 THEN
                        csgn = -csgn;
                  END;
               CALL cbinu(zn, fnu, kode, n, cy, nz, rl, fnul, tol, elim, alim);
               IF  nz >= 0  THEN
                  DO;
                     IF  xx >= 0.00000000000000000E0 THEN
                        RETURN;
/* ----------------------------------------------------------------------- */
/*     ANALYTIC CONTINUATION TO THE LEFT HALF PLANE                      */
/* ----------------------------------------------------------------------- */
                     nn = n - nz;
                     IF  nn  = 0 THEN
                        RETURN;
                     rtol = 1.00000000000000000E0 / tol;
                     ascle = TINY(0.00000000000000000E0) * rtol * 1.00000000000000000E+3;
                     DO  i = 1 TO  nn;
/*       CY(I) = CY(I)*CSGN                                              */
                        zn = cy(i);
                        aa = REAL(zn);
                        bb = IMAG(zn);
                        atol = 1;
                        IF  MAX(ABS(aa),ABS(bb)) <= ascle  THEN
                           DO;
                              zn = zn * rtol;
                              atol = tol;
                           END;
                        zn = zn * csgn;
                        cy(i) = zn * atol;
                        csgn = -csgn;
                     END;
                     RETURN;
                  END;
               IF  nz ^= -2  THEN
                  DO;
                     nz = 0;
                     ierr = 2;
                     RETURN;
                  END;
               nz = 0;
               ierr = 5;
               RETURN;
            END;
      END;
   nz = 0;
   ierr = 4;
   RETURN;
   END cbesi;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbesj: PROCEDURE (z, fnu, kode, n, cy, nz, ierr) OPTIONS (REORDER);             
/* ***BEGIN PROLOGUE  CBESJ                                              */
/* ***DATE WRITTEN   830501   (YYMMDD)                                   */
/* ***REVISION DATE  890801, 930101   (YYMMDD)                           */
/* ***CATEGORY NO.  B5K                                                  */
/* ***KEYWORDS  J-BESSEL FUNCTION,BESSEL FUNCTION OF COMPLEX ARGUMENT,   */
/*             BESSEL FUNCTION OF FIRST KIND                             */
/* ***AUTHOR  AMOS, DONALD E., SANDIA NATIONAL LABORATORIES              */
/* ***PURPOSE  TO COMPUTE THE J-BESSEL FUNCTION OF A COMPLEX ARGUMENT    */
/* ***DESCRIPTION                                                        */

/*    ON KODE=1, CBESJ COMPUTES AN N MEMBER  SEQUENCE OF COMPLEX         */
/*    BESSEL FUNCTIONS CY(I) = J(FNU+I-1,Z) FOR REAL (dp), NONNEGATIVE   */
/*    ORDERS FNU+I-1, I=1,...,N AND COMPLEX Z IN THE CUT PLANE           */
/*    -PI < ARG(Z) <= PI.  ON KODE=2, CBESJ RETURNS THE SCALED FUNCTIONS */

/*    CY(I) = EXP(-ABS(Y))*J(FNU+I-1,Z)   I = 1,...,N , Y=AIMAG(Z)       */

/*    WHICH REMOVE THE EXPONENTIAL GROWTH IN BOTH THE UPPER AND          */
/*    LOWER HALF PLANES FOR Z TO INFINITY.  DEFINITIONS AND NOTATION     */
/*    ARE FOUND IN THE NBS HANDBOOK OF MATHEMATICAL FUNCTIONS (REF. 1).  */

/*    INPUT                                                              */
/*      Z      - Z=CMPLX(X,Y),  -PI < ARG(Z) <= PI                       */
/*      FNU    - ORDER OF INITIAL J FUNCTION, FNU >= 0.0                 */
/*      KODE   - A PARAMETER TO INDICATE THE SCALING OPTION              */
/*               KODE= 1  RETURNS                                        */
/*                        CY(I)=J(FNU+I-1,Z), I=1,...,N                  */
/*                   = 2  RETURNS                                        */
/*                        CY(I)=J(FNU+I-1,Z)*EXP(-ABS(Y)), I=1,...       */
/*      N      - NUMBER OF MEMBERS OF THE SEQUENCE, N >= 1               */

/*    OUTPUT                                                             */
/*      CY     - A COMPLEX VECTOR WHOSE FIRST N COMPONENTS CONTAIN       */
/*               VALUES FOR THE SEQUENCE                                 */
/*               CY(I)=J(FNU+I-1,Z)  OR                                  */
/*               CY(I)=J(FNU+I-1,Z)*EXP(-ABS(Y))  I=1,...,N              */
/*               DEPENDING ON KODE, Y=AIMAG(Z).                          */
/*      NZ     - NUMBER OF COMPONENTS SET TO ZERO DUE TO UNDERFLOW,      */
/*               NZ= 0   , NORMAL RETURN                                 */
/*               NZ > 0 , LAST NZ COMPONENTS OF CY SET TO ZERO           */
/*                         DUE TO UNDERFLOW, CY(I)=CMPLX(0.0,0.0),       */
/*                         I = N-NZ+1,...,N                              */
/*      IERR   - ERROR FLAG                                              */
/*               IERR=0, NORMAL RETURN - COMPUTATION COMPLETED           */
/*               IERR=1, INPUT ERROR   - NO COMPUTATION                  */
/*               IERR=2, OVERFLOW      - NO COMPUTATION, AIMAG(Z)        */
/*                       TOO LARGE ON KODE=1                             */
/*               IERR=3, ABS(Z) OR FNU+N-1 LARGE - COMPUTATION DONE      */
/*                       BUT LOSSES OF SIGNIFICANCE BY ARGUMENT          */
/*                       REDUCTION PRODUCE LESS THAN HALF OF MACHINE ACCURACY */
/*               IERR=4, ABS(Z) OR FNU+N-1 TOO LARGE - NO COMPUTATION BECAUSE */
/*                       OF COMPLETE LOSSES OF SIGNIFICANCE BY ARGUMENT  */
/*                       REDUCTION                                       */
/*               IERR=5, ERROR              - NO COMPUTATION,            */
/*                       ALGORITHM TERMINATION CONDITION NOT MET         */

/* ***LONG DESCRIPTION                                                   */

/*    THE COMPUTATION IS CARRIED OUT BY THE FORMULA                      */

/*    J(FNU,Z)=EXP( FNU*PI*I/2)*I(FNU,-I*Z)    AIMAG(Z) >= 0.0           */

/*    J(FNU,Z)=EXP(-FNU*PI*I/2)*I(FNU, I*Z)    AIMAG(Z) < 0.0            */

/*    WHERE I**2 = -1 AND I(FNU,Z) IS THE I BESSEL FUNCTION.             */

/*    FOR NEGATIVE ORDERS,THE FORMULA                                    */

/*         J(-FNU,Z) = J(FNU,Z)*COS(PI*FNU) - Y(FNU,Z)*SIN(PI*FNU)       */

/*    CAN BE USED.  HOWEVER,FOR LARGE ORDERS CLOSE TO INTEGERS, THE FUNCTION */
/*    CHANGES RADICALLY.  WHEN FNU IS A LARGE POSITIVE INTEGER, THE MAGNITUDE */
/*    OF J(-FNU,Z)=J(FNU,Z)*COS(PI*FNU) IS A LARGE NEGATIVE POWER OF TEN. */
/*    BUT WHEN FNU IS NOT AN INTEGER, Y(FNU,Z) DOMINATES IN MAGNITUDE WITH A */
/*    LARGE POSITIVE POWER OF TEN AND THE MOST THAT THE SECOND TERM CAN BE */
/*    REDUCED IS BY UNIT ROUNDOFF FROM THE COEFFICIENT.  THUS, WIDE CHANGES CAN */
/*    OCCUR WITHIN UNIT ROUNDOFF OF A LARGE INTEGER FOR FNU.  HERE, LARGE MEANS */
/*    FNU > ABS(Z).                                                      */

/*    IN MOST COMPLEX VARIABLE COMPUTATION, ONE MUST EVALUATE ELEMENTARY */
/*    FUNCTIONS.  WHEN THE MAGNITUDE OF Z OR FNU+N-1 IS LARGE, LOSSES OF */
/*    SIGNIFICANCE BY ARGUMENT REDUCTION OCCUR.  CONSEQUENTLY, IF EITHER ONE */
/*    EXCEEDS U1=SQRT(0.5/UR), THEN LOSSES EXCEEDING HALF PRECISION ARE LIKELY */
/*    AND AN ERROR FLAG IERR=3 IS TRIGGERED WHERE UR = EPSILON(0.0) = UNIT */
/*    ROUNDOFF.  ALSO IF EITHER IS LARGER THAN U2=0.5/UR, THEN ALL SIGNIFICANCE */
/*    IS LOST AND IERR=4.  IN ORDER TO USE THE INT FUNCTION, ARGUMENTS MUST BE */
/*    FURTHER RESTRICTED NOT TO EXCEED THE LARGEST MACHINE INTEGER, U3 = HUGE(0). */
/*    THUS, THE MAGNITUDE OF Z AND FNU+N-1 IS RESTRICTED BY MIN(U2,U3).  */
/*    ON 32 BIT MACHINES, U1,U2, AND U3 ARE APPROXIMATELY 2.0E+3, 4.2E+6, 2.1E+9 */
/*    IN SINGLE PRECISION ARITHMETIC AND 1.3E+8, 1.8E+16, 2.1E+9 IN DOUBLE */
/*    PRECISION ARITHMETIC RESPECTIVELY.  THIS MAKES U2 AND U3 LIMITING IN */
/*    THEIR RESPECTIVE ARITHMETICS.  THIS MEANS THAT ONE CAN EXPECT TO RETAIN, */
/*    IN THE WORST CASES ON 32 BIT MACHINES, NO DIGITS IN SINGLE AND ONLY 7 */
/*    DIGITS IN DOUBLE PRECISION ARITHMETIC.                             */
/*    SIMILAR CONSIDERATIONS HOLD FOR OTHER MACHINES.                    */

/*    THE APPROXIMATE RELATIVE ERROR IN THE MAGNITUDE OF A COMPLEX BESSEL */
/*    FUNCTION CAN BE EXPRESSED BY P*10**S WHERE P = MAX(UNIT ROUNDOFF, 1.0E-18) */
/*    IS THE NOMINAL PRECISION AND 10**S REPRESENTS THE INCREASE IN ERROR DUE */
/*    TO ARGUMENT REDUCTION IN THE ELEMENTARY FUNCTIONS.  HERE,          */
/*    S = MAX(1,ABS(LOG10(ABS(Z))), ABS(LOG10(FNU))) APPROXIMATELY       */
/*    (I.E. S = MAX(1, ABS(EXPONENT OF ABS(Z), ABS(EXPONENT OF FNU)) ).  */
/*    HOWEVER, THE PHASE ANGLE MAY HAVE ONLY ABSOLUTE ACCURACY.  THIS IS MOST */
/*    LIKELY TO OCCUR WHEN ONE COMPONENT (IN ABSOLUTE VALUE) IS LARGER THAN */
/*    THE OTHER BY SEVERAL ORDERS OF MAGNITUDE.  IF ONE COMPONENT IS 10**K */
/*    LARGER THAN THE OTHER, THEN ONE CAN EXPECT ONLY MAX(ABS(LOG10(P))-K, 0) */
/*    SIGNIFICANT DIGITS; OR, STATED ANOTHER WAY, WHEN K EXCEEDS THE EXPONENT */
/*    OF P, NO SIGNIFICANT DIGITS REMAIN IN THE SMALLER COMPONENT.       */
/*    HOWEVER, THE PHASE ANGLE RETAINS ABSOLUTE ACCURACY BECAUSE, IN COMPLEX */
/*    ARITHMETIC WITH PRECISION P, THE SMALLER COMPONENT WILL NOT (AS A RULE) */
/*    DECREASE BELOW P TIMES THE MAGNITUDE OF THE LARGER COMPONENT.      */
/*    IN THESE EXTREME CASES, THE PRINCIPAL PHASE ANGLE IS ON THE ORDER OF +P, */
/*    -P, PI/2-P, OR -PI/2+P.                                            */

/* ***REFERENCES  HANDBOOK OF MATHEMATICAL FUNCTIONS BY M. ABRAMOWITZ AND */
/*         I. A. STEGUN, NBS AMS SERIES 55, U.S. DEPT. OF COMMERCE, 1955. */

/*       COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT             */
/*         BY D. E. AMOS, SAND83-0083, MAY 1983.                         */

/*       COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT             */
/*         AND LARGE ORDER BY D. E. AMOS, SAND83-0643, MAY 1983          */

/*       A SUBROUTINE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT */
/*         AND NONNEGATIVE ORDER BY D. E. AMOS, SAND85-1018, MAY 1985    */

/*       A PORTABLE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT   */
/*         AND NONNEGATIVE ORDER BY D. E. AMOS, ACM TRANS. MATH. SOFTWARE, */
/*         VOL. 12, NO. 3, SEPTEMBER 1986, PP 265-273.                   */

/* ***ROUTINES CALLED  CBINU,I1MACH,R1MACH                               */
/* ***END PROLOGUE  CBESJ                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( cy(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( ierr )  FIXED BINARY (31);

   DECLARE ( ci, csgn, zn )  COMPLEX FLOAT (18);
   DECLARE ( aa, alim, arg, dig, elim, fnul, rl, r1, r1m5, r2, tol, yy, az, 
      fn, bb, ascle, rtol, atol )  FLOAT (18);
   DECLARE ( i, inu, inuh, ir, k1, k2, nl, k )  FIXED BINARY (31);

   DECLARE ( hpi  STATIC INITIAL ( 1.570796326794896619E0) )  FLOAT (18);

/* ***FIRST EXECUTABLE STATEMENT  CBESJ                                  */
   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, inu, inuh );
      PUT SKIP DATA ( ir, k1, k2 );
      PUT SKIP DATA ( nl, k );
      RESIGNAL;
   END;

   ierr = 0;
   nz = 0;
   IF  fnu < 0 THEN
      ierr = 1;
   IF  kode < 1  |   kode > 2 THEN
      ierr = 1;
   IF  n < 1 THEN
      ierr = 1;
   IF  ierr ^= 0 THEN
      RETURN;
/* ----------------------------------------------------------------------- */
/*     SET PARAMETERS RELATED TO MACHINE CONSTANTS.                      */
/*     TOL IS THE APPROXIMATE UNIT ROUNDOFF LIMITED TO 1.0E-18.          */
/*     ELIM IS THE APPROXIMATE EXPONENTIAL OVER- AND UNDERFLOW LIMIT.    */
/*     EXP(-ELIM) < EXP(-ALIM)=EXP(-ELIM)/TOL    AND                     */
/*     EXP(ELIM) > EXP(ALIM)=EXP(ELIM)*TOL       ARE INTERVALS NEAR      */
/*     UNDERFLOW AND OVERFLOW LIMITS WHERE SCALED ARITHMETIC IS DONE.    */
/*     RL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC EXPANSION FOR LARGE Z. */
/*     DIG = NUMBER OF BASE 10 DIGITS IN TOL = 10**(-DIG).               */
/*     FNUL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC SERIES FOR LARGE FNU. */
/* ----------------------------------------------------------------------- */
   tol = MAX(EPSILON(0.00000000000000000E0), 1.00000000000000000E-18);
   k1 = MINEXPONENT(0.00000000000000000E0);
   k2 = MAXEXPONENT(0.00000000000000000E0);
   r1m5 = LOG10( RADIX(0.00000000000000000E0) );
   k = MIN(ABS(k1),ABS(k2));
   elim = 2.30300000000000000E0 * (k*r1m5 - 3.00000000000000000E0);
   k1 = DIGITS(0.00000000000000000E0) - 1;
   aa = r1m5 * k1;
   dig = MIN(aa, 18.0000000000000000E0);
   aa = aa * 2.30300000000000000E0;
   alim = elim + MAX(-aa, -41.4500000000000000E0);
   rl = 1.20000000000000000E0 * dig + 3.00000000000000000E0;
   fnul = 10.0000000000000000E0 + 6.00000000000000000E0 * (dig - 3.00000000000000000E0); 
       
   ci = 0.00000000000000000E0+1.00000000000000000E0i;
   yy = IMAG(z);
   az = ABS(z);
/* ----------------------------------------------------------------------- */
/*     TEST FOR RANGE                                                    */
/* ----------------------------------------------------------------------- */
   aa = 0.50000000000000000E0 / tol;
   bb = HUGE(0) * 0.50000000000000000E0;
   aa = MIN(aa,bb);
   fn = fnu + (n-1);
   IF  az <= aa  THEN
      DO;
         IF  fn <= aa  THEN
            DO;
               aa = SQRT(aa);
               IF  az > aa THEN
                  ierr = 3;
               IF  fn > aa THEN
                  ierr = 3;
/* ----------------------------------------------------------------------- */
/*     CALCULATE CSGN=EXP(FNU*HPI*I) TO MINIMIZE LOSSES OF SIGNIFICANCE  */
/*     WHEN FNU IS LARGE                                                 */
/* ----------------------------------------------------------------------- */
               inu = fnu;
               inuh = inu / 2;
               ir = inu - 2 * inuh;
               arg = (fnu - (inu-ir)) * hpi;
               r1 = COS(arg);
               r2 = SIN(arg);
               csgn = COMPLEX(r1, r2);
               IF  REM(inuh,2)  = 1 THEN
                  csgn = -csgn;
/* ----------------------------------------------------------------------- */
/*     ZN IS IN THE RIGHT HALF PLANE                                     */
/* ----------------------------------------------------------------------- */
               zn = -z * ci;
               IF  yy < 0  THEN
                  DO;
                     zn = -zn;
                     csgn = CONJG(csgn);
                     ci = CONJG(ci);
                  END;
               CALL cbinu(zn, fnu, kode, n, cy, nz, rl, fnul, tol, elim, alim);
               IF  nz >= 0  THEN
                  DO;
                     nl = n - nz;
                     IF  nl  = 0 THEN
                        RETURN;
                     rtol = 1.00000000000000000E0 / tol;
                     ascle = TINY(0.00000000000000000E0) * rtol * 1.00000000000000000E+3;
                     DO  i = 1 TO  nl;
/*       CY(I)=CY(I)*CSGN                                                */
                        zn = cy(i);
                        aa = REAL(zn);
                        bb = IMAG(zn);
                        atol = 1;
                        IF  MAX(ABS(aa),ABS(bb)) <= ascle  THEN
                           DO;
                              zn = zn * rtol;
                              atol = tol;
                           END;
                        zn = zn * csgn;
                        cy(i) = zn * atol;
                        csgn = csgn * ci;
                     END;
                     RETURN;
                  END;
               IF  nz ^= -2  THEN
                  DO;
                     nz = 0;
                     ierr = 2;
                     RETURN;
                  END;
               nz = 0;
               ierr = 5;
               RETURN;
            END;
      END;
   nz = 0;
   ierr = 4;
   RETURN;
   END cbesj;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbesk: PROCEDURE (z, fnu, kode, n, cy, nz, ierr) OPTIONS (REORDER);             
/* ***BEGIN PROLOGUE  CBESK                                              */
/* ***DATE WRITTEN   830501   (YYMMDD)                                   */
/* ***REVISION DATE  890801, 930101   (YYMMDD)                           */
/* ***CATEGORY NO.  B5K                                                  */
/* ***KEYWORDS  K-BESSEL FUNCTION,COMPLEX BESSEL FUNCTION,               */
/*             MODIFIED BESSEL FUNCTION OF THE SECOND KIND,              */
/*             BESSEL FUNCTION OF THE THIRD KIND                         */
/* ***AUTHOR  AMOS, DONALD E., SANDIA NATIONAL LABORATORIES              */
/* ***PURPOSE  TO COMPUTE K-BESSEL FUNCTIONS OF COMPLEX ARGUMENT         */
/* ***DESCRIPTION                                                        */

/*   ON KODE=1, CBESK COMPUTES AN N MEMBER SEQUENCE OF COMPLEX BESSEL FUNCTIONS */
/*   CY(J)=K(FNU+J-1,Z) FOR REAL (dp), NONNEGATIVE ORDERS FNU+J-1, J=1,...,N */
/*   AND COMPLEX Z.NE.CMPLX(0.0,0.0) IN THE CUT PLANE -PI < ARG(Z) <= PI. */
/*   ON KODE=2, CBESK RETURNS THE SCALED K FUNCTIONS,                    */

/*   CY(J)=EXP(Z)*K(FNU+J-1,Z) , J=1,...,N,                              */

/*   WHICH REMOVE THE EXPONENTIAL BEHAVIOR IN BOTH THE LEFT AND RIGHT HALF */
/*   PLANES FOR Z TO INFINITY.  DEFINITIONS AND NOTATION ARE FOUND IN THE NBS */
/*   HANDBOOK OF MATHEMATICAL FUNCTIONS (REF. 1).                        */

/*   INPUT                                                               */
/*     Z      - Z=CMPLX(X,Y),Z.NE.CMPLX(0.,0.),-PI < ARG(Z) <= PI        */
/*     FNU    - ORDER OF INITIAL K FUNCTION, FNU >= 0.0                  */
/*     N      - NUMBER OF MEMBERS OF THE SEQUENCE, N >= 1                */
/*     KODE   - A PARAMETER TO INDICATE THE SCALING OPTION               */
/*              KODE= 1  RETURNS                                         */
/*                       CY(I)=K(FNU+I-1,Z), I=1,...,N                   */
/*                  = 2  RETURNS                                         */
/*                       CY(I)=K(FNU+I-1,Z)*EXP(Z), I=1,...,N            */

/*   OUTPUT                                                              */
/*     CY     - A COMPLEX VECTOR WHOSE FIRST N COMPONENTS CONTAIN        */
/*              VALUES FOR THE SEQUENCE                                  */
/*              CY(I)=K(FNU+I-1,Z), I=1,...,N OR                         */
/*              CY(I)=K(FNU+I-1,Z)*EXP(Z), I=1,...,N                     */
/*              DEPENDING ON KODE                                        */
/*     NZ     - NUMBER OF COMPONENTS SET TO ZERO DUE TO UNDERFLOW.       */
/*              NZ= 0   , NORMAL RETURN                                  */
/*              NZ > 0 , FIRST NZ COMPONENTS OF CY SET TO ZERO           */
/*                        DUE TO UNDERFLOW, CY(I)=CMPLX(0.0,0.0),        */
/*                        I=1,...,N WHEN X >= 0.0.  WHEN X < 0.0, NZ STATES */
/*                        ONLY THE NUMBER OF UNDERFLOWS IN THE SEQUENCE. */
/*     IERR   - ERROR FLAG                                               */
/*              IERR=0, NORMAL RETURN - COMPUTATION COMPLETED            */
/*              IERR=1, INPUT ERROR   - NO COMPUTATION                   */
/*              IERR=2, OVERFLOW      - NO COMPUTATION, FNU+N-1 IS       */
/*                      TOO LARGE OR ABS(Z) IS TOO SMALL OR BOTH         */
/*              IERR=3, ABS(Z) OR FNU+N-1 LARGE - COMPUTATION DONE, BUT  */
/*                      LOSSES OF SIGNIFICANCE BY ARGUMENT REDUCTION PRODUCE */
/*                      LESS THAN HALF OF MACHINE ACCURACY               */
/*              IERR=4, ABS(Z) OR FNU+N-1 TOO LARGE - NO COMPUTATION BECAUSE OF */
/*                      COMPLETE LOSSES OF SIGNIFICANCE BY ARGUMENT REDUCTION */
/*              IERR=5, ERROR              - NO COMPUTATION,             */
/*                      ALGORITHM TERMINATION CONDITION NOT MET          */

/* ***LONG DESCRIPTION                                                   */

/*   EQUATIONS OF THE REFERENCE ARE IMPLEMENTED FOR SMALL ORDERS DNU AND */
/*   DNU+1.0 IN THE RIGHT HALF PLANE X >= 0.0.  FORWARD RECURRENCE GENERATES */
/*   HIGHER ORDERS.  K IS CONTINUED TO THE LEFT HALF PLANE BY THE RELATION */

/*   K(FNU,Z*EXP(MP)) = EXP(-MP*FNU)*K(FNU,Z)-MP*I(FNU,Z)                */
/*   MP=MR*PI*I, MR=+1 OR -1, RE(Z) > 0, I**2=-1                         */

/*   WHERE I(FNU,Z) IS THE I BESSEL FUNCTION.                            */

/*   FOR LARGE ORDERS, FNU > FNUL, THE K FUNCTION IS COMPUTED            */
/*   BY MEANS OF ITS UNIFORM ASYMPTOTIC EXPANSIONS.                      */

/*   FOR NEGATIVE ORDERS, THE FORMULA                                    */

/*                 K(-FNU,Z) = K(FNU,Z)                                  */

/*   CAN BE USED.                                                        */

/*   CBESK ASSUMES THAT A SIGNIFICANT DIGIT SINH(X) FUNCTION IS AVAILABLE. */

/*   IN MOST COMPLEX VARIABLE COMPUTATION, ONE MUST EVALUATE ELEMENTARY  */
/*   FUNCTIONS.  WHEN THE MAGNITUDE OF Z OR FNU+N-1 IS LARGE, LOSSES OF  */
/*   SIGNIFICANCE BY ARGUMENT REDUCTION OCCUR.                           */
/*   CONSEQUENTLY, IF EITHER ONE EXCEEDS U1=SQRT(0.5/UR), THEN LOSSES EXCEEDING */
/*   HALF PRECISION ARE LIKELY AND AN ERROR FLAG IERR=3 IS TRIGGERED WHERE */
/*   UR = EPSILON(0.0) = UNIT ROUNDOFF.  ALSO IF EITHER IS LARGER THAN   */
/*   U2 = 0.5/UR, THEN ALL SIGNIFICANCE IS LOST AND IERR=4.  IN ORDER TO USE */
/*   THE INT FUNCTION, ARGUMENTS MUST BE FURTHER RESTRICTED NOT TO EXCEED THE */
/*   LARGEST MACHINE INTEGER, U3=HUGE(0).  THUS, THE MAGNITUDE OF Z AND FNU+N-1 */
/*   IS RESTRICTED BY MIN(U2,U3).  ON 32 BIT MACHINES, U1,U2, AND U3 ARE */
/*   APPROXIMATELY 2.0E+3, 4.2E+6, 2.1E+9 IN SINGLE PRECISION ARITHMETIC AND */
/*   1.3E+8, 1.8E+16, 2.1E+9 IN DOUBLE PRECISION ARITHMETIC RESPECTIVELY. */
/*   THIS MAKES U2 AND U3 LIMITING IN THEIR RESPECTIVE ARITHMETICS.  THIS MEANS */
/*   THAT ONE CAN EXPECT TO RETAIN, IN THE WORST CASES ON 32 BIT MACHINES, NO */
/*   DIGITS IN SINGLE AND ONLY 7 DIGITS IN DOUBLE PRECISION ARITHMETIC.  */
/*   SIMILAR CONSIDERATIONS HOLD FOR OTHER MACHINES.                     */

/*   THE APPROXIMATE RELATIVE ERROR IN THE MAGNITUDE OF A COMPLEX BESSEL */
/*   FUNCTION CAN BE EXPRESSED BY P*10**S WHERE P = MAX(UNIT ROUNDOFF,1.0E-18) */
/*   IS THE NOMINAL PRECISION AND 10**S REPRESENTS THE INCREASE IN ERROR DUE TO */
/*   ARGUMENT REDUCTION IN THE ELEMENTARY FUNCTIONS.  HERE, S =          */
/*   MAX(1,ABS(LOG10(ABS(Z))), ABS(LOG10(FNU))) APPROXIMATELY (I.E. S =  */
/*   MAX(1,ABS(EXPONENT OF ABS(Z),ABS(EXPONENT OF FNU)) ).               */
/*   HOWEVER, THE PHASE ANGLE MAY HAVE ONLY ABSOLUTE ACCURACY.  THIS IS MOST */
/*   LIKELY TO OCCUR WHEN ONE COMPONENT (IN ABSOLUTE VALUE) IS LARGER THAN THE */
/*   OTHER BY SEVERAL ORDERS OF MAGNITUDE.  IF ONE COMPONENT IS 10**K LARGER */
/*   THAN THE OTHER, THEN ONE CAN EXPECT ONLY MAX(ABS(LOG10(P))-K, 0)    */
/*   SIGNIFICANT DIGITS; OR, STATED ANOTHER WAY, WHEN K EXCEEDS THE EXPONENT */
/*   OF P, NO SIGNIFICANT DIGITS REMAIN IN THE SMALLER COMPONENT.  HOWEVER, THE */
/*   PHASE ANGLE RETAINS ABSOLUTE ACCURACY BECAUSE, IN COMPLEX ARITHMETIC WITH */
/*   PRECISION P, THE SMALLER COMPONENT WILL NOT (AS A RULE) DECREASE BELOW P */
/*   TIMES THE MAGNITUDE OF THE LARGER COMPONENT.  IN THESE EXTREME CASES, */
/*   THE PRINCIPAL PHASE ANGLE IS ON THE ORDER OF +P, -P, PI/2-P, OR -PI/2+P. */

/* ***REFERENCES  HANDBOOK OF MATHEMATICAL FUNCTIONS BY M. ABRAMOWITZ AND */
/*           I. A. STEGUN, NBS AMS SERIES 55, U.S. DEPT. OF COMMERCE, 1955. */

/*         COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT           */
/*           BY D. E. AMOS, SAND83-0083, MAY 1983.                       */

/*         COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT           */
/*           AND LARGE ORDER BY D. E. AMOS, SAND83-0643, MAY 1983.       */

/*         A SUBROUTINE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT */
/*           AND NONNEGATIVE ORDER BY D. E. AMOS, SAND85-1018, MAY 1985  */

/*         A PORTABLE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT */
/*           AND NONNEGATIVE ORDER BY D. E. AMOS, ACM TRANS. MATH. SOFTWARE, */
/*           VOL. 12, NO. 3, SEPTEMBER 1986, PP 265-273.                 */

/* ***ROUTINES CALLED  CACON,CBKNU,CBUNK,CUOIK,I1MACH,R1MACH             */
/* ***END PROLOGUE  CBESK                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( cy(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( ierr )  FIXED BINARY (31);

   DECLARE ( aa, alim, aln, arg, az, dig, elim, fn, fnul, rl, r1m5, tol, ufl, 
      xx, yy, bb )  FLOAT (18);
   DECLARE ( k, k1, k2, mr, nn, nuf, nw )  FIXED BINARY (31);

/* ***FIRST EXECUTABLE STATEMENT  CBESK                                  */
   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( k, k1, k2 );
      PUT SKIP DATA ( mr, nn, nuf, nw );
      RESIGNAL;
   END;

   ierr = 0;
   nz = 0;
   xx = REAL(z);
   yy = IMAG(z);
   IF  yy  = 0   &   xx  = 0 THEN
      ierr = 1;
   IF  fnu < 0 THEN
      ierr = 1;
   IF  kode < 1  |   kode > 2 THEN
      ierr = 1;
   IF  n < 1 THEN
      ierr = 1;
   IF  ierr ^= 0 THEN
      RETURN;
   nn = n;
/* ----------------------------------------------------------------------- */
/*     SET PARAMETERS RELATED TO MACHINE CONSTANTS.                      */
/*     TOL IS THE APPROXIMATE UNIT ROUNDOFF LIMITED TO 1.0E-18.          */
/*     ELIM IS THE APPROXIMATE EXPONENTIAL OVER- AND UNDERFLOW LIMIT.    */
/*     EXP(-ELIM) < EXP(-ALIM)=EXP(-ELIM)/TOL    AND                     */
/*     EXP(ELIM) > EXP(ALIM)=EXP(ELIM)*TOL       ARE INTERVALS NEAR      */
/*     UNDERFLOW AND OVERFLOW LIMITS WHERE SCALED ARITHMETIC IS DONE.    */
/*     RL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC EXPANSION FOR LARGE Z. */
/*     DIG = NUMBER OF BASE 10 DIGITS IN TOL = 10**(-DIG).               */
/*     FNUL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC SERIES FOR LARGE FNU */
/* ----------------------------------------------------------------------- */
   tol = MAX(EPSILON(0.00000000000000000E0), 1.00000000000000000E-18);
   k1 = MINEXPONENT(0.00000000000000000E0);
   k2 = MAXEXPONENT(0.00000000000000000E0);
   r1m5 = LOG10( RADIX(0.00000000000000000E0) );
   k = MIN(ABS(k1),ABS(k2));
   elim = 2.30300000000000000E0 * (k*r1m5 - 3.00000000000000000E0);
   k1 = DIGITS(0.00000000000000000E0) - 1;
   aa = r1m5 * k1;
   dig = MIN(aa, 18.0000000000000000E0);
   aa = aa * 2.30300000000000000E0;
   alim = elim + MAX(-aa, -41.4500000000000000E0);
   fnul = 10.0000000000000000E0 + 6.00000000000000000E0 * (dig - 3.00000000000000000E0); 
       
   rl = 1.20000000000000000E0 * dig + 3.00000000000000000E0;
   az = ABS(z);
   fn = fnu + (nn-1);
/* ----------------------------------------------------------------------- */
/*     TEST FOR RANGE                                                    */
/* ----------------------------------------------------------------------- */
   aa = 0.50000000000000000E0 / tol;
   bb = HUGE(0) * 0.50000000000000000E0;
   aa = MIN(aa,bb);
   IF  az <= aa  THEN
      DO;
         IF  fn <= aa  THEN
            DO;
               aa = SQRT(aa);
               IF  az > aa THEN
                  ierr = 3;
               IF  fn > aa THEN
                  ierr = 3;
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST ON THE LAST MEMBER OF THE SEQUENCE                  */
/* ----------------------------------------------------------------------- */
/*     UFL = EXP(-ELIM)                                                  */
               ufl = TINY(0.00000000000000000E0) * 1.00000000000000000E+3;
               IF  az >= ufl  THEN
                  DO;
                     IF  fnu <= fnul  THEN
                        DO;
                           IF  fn > 1  THEN
                              DO;
                                 IF  fn <= 2  THEN
                                    DO;
                                       IF  az > tol THEN  GO TO L10;
                                       arg = 0.50000000000000000E0 * az;
                                       aln = -fn * LOG(arg);
                                       IF  aln > elim THEN  GO TO L30;
                                    END;
                                 ELSE
                                    DO;
                                       CALL cuoik(z, fnu, kode, 2, nn, cy, nuf, tol, elim, alim);
                                       IF  nuf < 0 THEN  GO TO L30;
                                       nz = nz + nuf;
                                       nn = nn - nuf;
/* ----------------------------------------------------------------------- */
/*     HERE NN=N OR NN=0 SINCE NUF=0,NN, OR -1 ON RETURN FROM CUOIK      */
/*     IF NUF=NN, THEN CY(I)=CZERO FOR ALL I                             */
/* ----------------------------------------------------------------------- */
                                       IF  nn  = 0 THEN  GO TO L20;
                                    END;
                              END;

L10:
                           IF  xx >= 0  THEN
                              DO;
/* ----------------------------------------------------------------------- */
/*     RIGHT HALF PLANE COMPUTATION, REAL(Z) >= 0.                       */
/* ----------------------------------------------------------------------- */
                                 CALL cbknu(z, fnu, kode, nn, cy, nw, tol, elim, alim);
                                 IF  nw < 0 THEN  GO TO L40;
                                 nz = nw;
                                 RETURN;
                              END;
/* ----------------------------------------------------------------------- */
/*     LEFT HALF PLANE COMPUTATION                                       */
/*     PI/2 < ARG(Z) <= PI AND -PI < ARG(Z) < -PI/2.                     */
/* ----------------------------------------------------------------------- */
                           IF  nz ^= 0 THEN  GO TO L30;
                           mr = 1;
                           IF  yy < 0 THEN
                              mr = -1;
                           CALL cacon(z, fnu, kode, mr, nn, cy, nw, rl, fnul, tol, elim, alim);
                           IF  nw < 0 THEN  GO TO L40;
                           nz = nw;
                           RETURN;
                        END;
/* ----------------------------------------------------------------------- */
/*     UNIFORM ASYMPTOTIC EXPANSIONS FOR FNU > FNUL                      */
/* ----------------------------------------------------------------------- */
                     mr = 0;
                     IF  xx < 0  THEN
                        DO;
                           mr = 1;
                           IF  yy < 0 THEN
                              mr = -1;
                        END;
                     CALL cbunk(z, fnu, kode, mr, nn, cy, nw, tol, elim, alim);
                     IF  nw < 0 THEN  GO TO L40;
                     nz = nz + nw;
                     RETURN;

L20:
                     IF  xx >= 0 THEN
                        RETURN;
                  END;

L30:
               nz = 0;
               ierr = 2;
               RETURN;

L40:
               IF  nw  = -1 THEN  GO TO L30;
               nz = 0;
               ierr = 5;
               RETURN;
            END;
      END;
   nz = 0;
   ierr = 4;
   RETURN;
   END cbesk;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbesy: PROCEDURE (z, fnu, kode, n, cy, nz, ierr) OPTIONS (REORDER);             

/* N.B. Argument CWRK has been removed.                                  */

/* ***BEGIN PROLOGUE  CBESY                                              */
/* ***DATE WRITTEN   830501   (YYMMDD)                                   */
/* ***REVISION DATE  890801, 930101  (YYMMDD)                            */
/* ***CATEGORY NO.  B5K                                                  */
/* ***KEYWORDS  Y-BESSEL FUNCTION,BESSEL FUNCTION OF COMPLEX ARGUMENT,   */
/*             BESSEL FUNCTION OF SECOND KIND                            */
/* ***AUTHOR  AMOS, DONALD E., SANDIA NATIONAL LABORATORIES              */
/* ***PURPOSE  TO COMPUTE THE Y-BESSEL FUNCTION OF A COMPLEX ARGUMENT    */
/* ***DESCRIPTION                                                        */

/*   ON KODE=1, CBESY COMPUTES AN N MEMBER SEQUENCE OF COMPLEX           */
/*   BESSEL FUNCTIONS CY(I)=Y(FNU+I-1,Z) FOR REAL (dp), NONNEGATIVE      */
/*   ORDERS FNU+I-1, I=1,...,N AND COMPLEX Z IN THE CUT PLANE            */
/*   -PI < ARG(Z) <= PI.                                                 */
/*   ON KODE=2, CBESY RETURNS THE SCALED FUNCTIONS                       */

/*   CY(I)=EXP(-ABS(Y))*Y(FNU+I-1,Z)   I = 1,...,N , Y=AIMAG(Z)          */

/*   WHICH REMOVE THE EXPONENTIAL GROWTH IN BOTH THE UPPER AND           */
/*   LOWER HALF PLANES FOR Z TO INFINITY. DEFINITIONS AND NOTATION       */
/*   ARE FOUND IN THE NBS HANDBOOK OF MATHEMATICAL FUNCTIONS (REF. 1).   */

/*   INPUT                                                               */
/*     Z      - Z=CMPLX(X,Y), Z.NE.CMPLX(0.,0.),-PI < ARG(Z) <= PI       */
/*     FNU    - ORDER OF INITIAL Y FUNCTION, FNU >= 0.0                  */
/*     KODE   - A PARAMETER TO INDICATE THE SCALING OPTION               */
/*              KODE= 1  RETURNS                                         */
/*                       CY(I)=Y(FNU+I-1,Z), I=1,...,N                   */
/*                  = 2  RETURNS                                         */
/*                       CY(I)=Y(FNU+I-1,Z)*EXP(-ABS(Y)), I=1,...,N      */
/*                       WHERE Y=AIMAG(Z)                                */
/*     N      - NUMBER OF MEMBERS OF THE SEQUENCE, N >= 1                */
/*     CWRK   - A COMPLEX WORK VECTOR OF DIMENSION AT LEAST N            */

/*   OUTPUT                                                              */
/*     CY     - A COMPLEX VECTOR WHOSE FIRST N COMPONENTS CONTAIN        */
/*              VALUES FOR THE SEQUENCE                                  */
/*              CY(I)=Y(FNU+I-1,Z)  OR                                   */
/*              CY(I)=Y(FNU+I-1,Z)*EXP(-ABS(Y))  I=1,...,N               */
/*              DEPENDING ON KODE.                                       */
/*     NZ     - NZ=0 , A NORMAL RETURN                                   */
/*              NZ > 0 , NZ COMPONENTS OF CY SET TO ZERO DUE TO          */
/*              UNDERFLOW (GENERALLY ON KODE=2)                          */
/*     IERR   - ERROR FLAG                                               */
/*              IERR=0, NORMAL RETURN - COMPUTATION COMPLETED            */
/*              IERR=1, INPUT ERROR   - NO COMPUTATION                   */
/*              IERR=2, OVERFLOW      - NO COMPUTATION, FNU+N-1 IS       */
/*                      TOO LARGE OR ABS(Z) IS TOO SMALL OR BOTH         */
/*              IERR=3, ABS(Z) OR FNU+N-1 LARGE - COMPUTATION DONE       */
/*                      BUT LOSSES OF SIGNIFICANCE BY ARGUMENT REDUCTION */
/*                      PRODUCE LESS THAN HALF OF MACHINE ACCURACY       */
/*              IERR=4, ABS(Z) OR FNU+N-1 TOO LARGE - NO COMPUTATION     */
/*                      BECAUSE OF COMPLETE LOSSES OF SIGNIFICANCE       */
/*                      BY ARGUMENT REDUCTION                            */
/*              IERR=5, ERROR              - NO COMPUTATION,             */
/*                      ALGORITHM TERMINATION CONDITION NOT MET          */

/* ***LONG DESCRIPTION                                                   */

/*   THE COMPUTATION IS CARRIED OUT IN TERMS OF THE I(FNU,Z) AND         */
/*   K(FNU,Z) BESSEL FUNCTIONS IN THE RIGHT HALF PLANE BY                */

/*       Y(FNU,Z) = I*CC*I(FNU,ARG) - (2/PI)*CONJG(CC)*K(FNU,ARG)        */

/*       Y(FNU,Z) = CONJG(Y(FNU,CONJG(Z)))                               */

/*   FOR AIMAG(Z) >= 0 AND AIMAG(Z) < 0 RESPECTIVELY, WHERE              */
/*   CC=EXP(I*PI*FNU/2), ARG=Z*EXP(-I*PI/2) AND I**2=-1.                 */

/*   FOR NEGATIVE ORDERS,THE FORMULA                                     */

/*       Y(-FNU,Z) = Y(FNU,Z)*COS(PI*FNU) + J(FNU,Z)*SIN(PI*FNU)         */

/*   CAN BE USED.  HOWEVER,FOR LARGE ORDERS CLOSE TO HALF ODD INTEGERS THE */
/*   FUNCTION CHANGES RADICALLY.  WHEN FNU IS A LARGE POSITIVE HALF ODD INTEGER, */
/*   THE MAGNITUDE OF Y(-FNU,Z) = J(FNU,Z)*SIN(PI*FNU) IS A LARGE NEGATIVE */
/*   POWER OF TEN.  BUT WHEN FNU IS NOT A HALF ODD INTEGER, Y(FNU,Z) DOMINATES */
/*   IN MAGNITUDE WITH A LARGE POSITIVE POWER OF TEN AND THE MOST THAT THE */
/*   SECOND TERM CAN BE REDUCED IS BY UNIT ROUNDOFF FROM THE COEFFICIENT. */
/*   THUS, WIDE CHANGES CAN OCCUR WITHIN UNIT ROUNDOFF OF A LARGE HALF   */
/*   ODD INTEGER.  HERE, LARGE MEANS FNU > ABS(Z).                       */

/*   IN MOST COMPLEX VARIABLE COMPUTATION, ONE MUST EVALUATE ELEMENTARY  */
/*   FUNCTIONS.  WHEN THE MAGNITUDE OF Z OR FNU+N-1 IS LARGE, LOSSES OF  */
/*   SIGNIFICANCE BY ARGUMENT REDUCTION OCCUR.  CONSEQUENTLY, IF EITHER ONE */
/*   EXCEEDS U1=SQRT(0.5/UR), THEN LOSSES EXCEEDING HALF PRECISION ARE LIKELY */
/*   AND AN ERROR FLAG IERR=3 IS TRIGGERED WHERE UR = EPSILON(0.0) = UNIT */
/*   ROUNDOFF.  ALSO IF EITHER IS LARGER THAN U2=0.5/UR, THEN ALL SIGNIFICANCE */
/*   IS LOST AND IERR=4.  IN ORDER TO USE THE INT FUNCTION, ARGUMENTS MUST BE */
/*   FURTHER RESTRICTED NOT TO EXCEED THE LARGEST MACHINE INTEGER, U3 = HUGE(0). */
/*   THUS, THE MAGNITUDE OF Z AND FNU+N-1 IS RESTRICTED BY MIN(U2,U3).   */
/*   ON 32 BIT MACHINES, U1,U2, AND U3 ARE APPROXIMATELY 2.0E+3, 4.2E+6, 2.1E+9 */
/*   IN SINGLE PRECISION ARITHMETIC AND 1.3E+8, 1.8E+16, 2.1E+9 IN DOUBLE */
/*   PRECISION ARITHMETIC RESPECTIVELY. THIS MAKES U2 AND U3 LIMITING IN */
/*   THEIR RESPECTIVE ARITHMETICS.  THIS MEANS THAT ONE CAN EXPECT TO RETAIN, */
/*   IN THE WORST CASES ON 32 BIT MACHINES, NO DIGITS IN SINGLE AND ONLY */
/*   7 DIGITS IN DOUBLE PRECISION ARITHMETIC.                            */
/*   SIMILAR CONSIDERATIONS HOLD FOR OTHER MACHINES.                     */

/*   THE APPROXIMATE RELATIVE ERROR IN THE MAGNITUDE OF A COMPLEX BESSEL */
/*   FUNCTION CAN BE EXPRESSED BY P*10**S WHERE P = MAX(UNIT ROUNDOFF,1.0E-18) */
/*   IS THE NOMINAL PRECISION AND 10**S REPRESENTS THE INCREASE IN ERROR DUE TO */
/*   ARGUMENT REDUCTION IN THE ELEMENTARY FUNCTIONS.  HERE, S =          */
/*   MAX(1,ABS(LOG10(ABS(Z))), ABS(LOG10(FNU))) APPROXIMATELY (I.E. S =  */
/*   MAX(1,ABS(EXPONENT OF ABS(Z),ABS(EXPONENT OF FNU)) ).               */
/*   HOWEVER, THE PHASE ANGLE MAY HAVE ONLY ABSOLUTE ACCURACY.  THIS IS MOST */
/*   LIKELY TO OCCUR WHEN ONE COMPONENT (IN ABSOLUTE VALUE) IS LARGER THAN THE */
/*   OTHER BY SEVERAL ORDERS OF MAGNITUDE.  IF ONE COMPONENT IS 10**K LARGER */
/*   THAN THE OTHER, THEN ONE CAN EXPECT ONLY MAX(ABS(LOG10(P))-K, 0)    */
/*   SIGNIFICANT DIGITS; OR, STATED ANOTHER WAY, WHEN K EXCEEDS THE EXPONENT */
/*   OF P, NO SIGNIFICANT DIGITS REMAIN IN THE SMALLER COMPONENT.        */
/*   HOWEVER, THE PHASE ANGLE RETAINS ABSOLUTE ACCURACY BECAUSE, IN COMPLEX */
/*   ARITHMETIC WITH PRECISION P, THE SMALLER COMPONENT WILL NOT (AS A RULE) */
/*   DECREASE BELOW P TIMES THE MAGNITUDE OF THE LARGER COMPONENT.  IN THESE */
/*   EXTREME CASES, THE PRINCIPAL PHASE ANGLE IS ON THE ORDER OF +P, -P, */
/*   PI/2-P, OR -PI/2+P.                                                 */

/* ***REFERENCES  HANDBOOK OF MATHEMATICAL FUNCTIONS BY M. ABRAMOWITZ AND */
/*        I. A. STEGUN, NBS AMS SERIES 55, U.S. DEPT. OF COMMERCE, 1955. */

/*      COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT              */
/*        BY D. E. AMOS, SAND83-0083, MAY 1983.                          */

/*      COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT              */
/*        AND LARGE ORDER BY D. E. AMOS, SAND83-0643, MAY 1983           */

/*      A SUBROUTINE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT  */
/*        AND NONNEGATIVE ORDER BY D. E. AMOS, SAND85-1018, MAY 1985     */

/*      A PORTABLE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT    */
/*        AND NONNEGATIVE ORDER BY D. E. AMOS, ACM TRANS. MATH. SOFTWARE, */
/*        VOL. 12, NO. 3, SEPTEMBER 1986, PP 265-273.                    */

/* ***ROUTINES CALLED  CBESI,CBESK,I1MACH,R1MACH                         */
/* ***END PROLOGUE  CBESY                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( cy(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( ierr )  FIXED BINARY (31);

   DECLARE ( ci, csgn, cspn, cwrk(n), ex, zu, zv, zz, zn )  COMPLEX FLOAT (18);
   DECLARE ( arg, elim, ey, r1, r2, tay, xx, yy, ascle, rtol, atol, tol, aa, 
      bb, ffnu, rhpi, r1m5 )  FLOAT (18);
   DECLARE ( i, ifnu, k, k1, k2, nz1, nz2, i4 )  FIXED BINARY (31);
   DECLARE ( cip(4)  STATIC INITIAL (
                    1.00000000000000000E0+0.00000000000000000E0i,
                    0.00000000000000000E0+1.00000000000000000E0i,
                   -1.00000000000000000E0+0.00000000000000000E0i,
                    0.00000000000000000E0-1.00000000000000000E0i ) )
                COMPLEX FLOAT (18);
   DECLARE ( hpi  STATIC INITIAL ( 1.57079632679489662E0) )  FLOAT (18);

/* ***FIRST EXECUTABLE STATEMENT  CBESY                                  */
   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, ifnu, k );
      PUT SKIP DATA ( k1, k2, nz1, nz2, i4 );
      RESIGNAL;
   END;

   xx = REAL(z);
   yy = IMAG(z);
   ierr = 0;
   nz = 0;
   IF  xx  = 0   &   yy  = 0 THEN
      ierr = 1;
   IF  fnu < 0 THEN
      ierr = 1;
   IF  kode < 1  |   kode > 2 THEN
      ierr = 1;
   IF  n < 1 THEN
      ierr = 1;
   IF  ierr ^= 0 THEN
      RETURN;
   ci = 0+1.00000000000000000E0i;
   zz = z;
   IF  yy < 0 THEN
      zz = CONJG(z);
   zn = -ci * zz;
   CALL cbesi(zn, fnu, kode, n, cy, nz1, ierr);
   IF  ierr  = 0  |   ierr  = 3  THEN
      DO;
         CALL cbesk(zn, fnu, kode, n, cwrk, nz2, ierr);
         IF  ierr  = 0  |   ierr  = 3  THEN
            DO;
               nz = MIN(nz1, nz2);
               ifnu = fnu;
               ffnu = fnu - ifnu;
               arg = hpi * ffnu;
               csgn = COMPLEX(COS(arg), SIN(arg) );
               i4 = REM(ifnu, 4) + 1;
               csgn = csgn * cip(i4);
               rhpi = 1.00000000000000000E0 / hpi;
               cspn = CONJG(csgn) * rhpi;
               csgn = csgn * ci;
               IF  kode ^= 2  THEN
                  DO;
                     DO  i = 1 TO  n;
                        cy(i) = csgn * cy(i) - cspn * cwrk(i);
                        csgn = ci * csgn;
                        cspn = -ci * cspn;
                     END;
                     IF  yy < 0 THEN
                        cy(*) = CONJG(cy(*));
                     RETURN;
                  END;

               r1 = COS(xx);
               r2 = SIN(xx);
               ex = COMPLEX(r1, r2 );
               tol = MAX(EPSILON(0.00000000000000000E0), 1.00000000000000000E-18);
               k1 = MINEXPONENT(0.00000000000000000E0);
               k2 = MAXEXPONENT(0.00000000000000000E0);
               k = MIN(ABS(k1),ABS(k2));
               r1m5 = LOG10( RADIX(0.00000000000000000E0) );
/* ----------------------------------------------------------------------- */
/*     ELIM IS THE APPROXIMATE EXPONENTIAL UNDER- AND OVERFLOW LIMIT     */
/* ----------------------------------------------------------------------- */
               elim = 2.30300000000000000E0 * (k*r1m5 - 3.00000000000000000E0);
               ey = 0.00000000000000000E0;
               tay = ABS(yy+yy);
               IF  tay < elim THEN
                  ey = EXP(-tay);
               cspn = ex * ey * cspn;
               nz = 0;
               rtol = 1.00000000000000000E0 / tol;
               ascle = TINY(0.00000000000000000E0) * rtol * 1.00000000000000000E+3;
               DO  i = 1 TO  n;
/* ---------------------------------------------------------------------- */
/*       CY(I) = CSGN*CY(I)-CSPN*CWRK(I): PRODUCTS ARE COMPUTED IN       */
/*       SCALED MODE IF CY(I) OR CWRK(I) ARE CLOSE TO UNDERFLOW TO       */
/*       PREVENT UNDERFLOW IN AN INTERMEDIATE COMPUTATION.               */
/* ---------------------------------------------------------------------- */
                  zv = cwrk(i);
                  aa = REAL(zv);
                  bb = IMAG(zv);
                  atol = 1.00000000000000000E0;
                  IF  MAX(ABS(aa),ABS(bb)) <= ascle  THEN
                     DO;
                        zv = zv * rtol;
                        atol = tol;
                     END;
                  zv = zv * cspn;
                  zv = zv * atol;
                  zu = cy(i);
                  aa = REAL(zu);
                  bb = IMAG(zu);
                  atol = 1.00000000000000000E0;
                  IF  MAX(ABS(aa),ABS(bb)) <= ascle  THEN
                     DO;
                        zu = zu * rtol;
                        atol = tol;
                     END;
                  zu = zu * csgn;
                  zu = zu * atol;
                  cy(i) = zu - zv;
                  IF  yy < 0 THEN
                     cy(i) = CONJG(cy(i));
                  IF  cy(i)  = 0.00000000000000000E0+0.00000000000000000E0i
                     &   ey  = 0 THEN
                     nz = nz + 1;
                  csgn =  ci * csgn;
                  cspn = -ci * cspn;
               END;
               RETURN;
            END;
      END;
   nz = 0;
   RETURN;
   END cbesy;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cairy: PROCEDURE (z, id, kode, ai, nz, ierr) OPTIONS (REORDER);                 
/* ***BEGIN PROLOGUE  CAIRY                                              */
/* ***DATE WRITTEN   830501   (YYMMDD)                                   */
/* ***REVISION DATE  890801, 930101   (YYMMDD)                           */
/* ***CATEGORY NO.  B5K                                                  */
/* ***KEYWORDS  AIRY FUNCTION,BESSEL FUNCTIONS OF ORDER ONE THIRD        */
/* ***AUTHOR  AMOS, DONALD E., SANDIA NATIONAL LABORATORIES              */
/* ***PURPOSE  TO COMPUTE AIRY FUNCTIONS AI(Z) AND DAI(Z) FOR COMPLEX Z  */
/* ***DESCRIPTION                                                        */

/*   ON KODE=1, CAIRY COMPUTES THE COMPLEX AIRY FUNCTION AI(Z) OR        */
/*   ITS DERIVATIVE DAI(Z)/DZ ON ID=0 OR ID=1 RESPECTIVELY.  ON          */
/*   KODE=2, A SCALING OPTION EXP(ZTA)*AI(Z) OR EXP(ZTA)*                */
/*   DAI(Z)/DZ IS PROVIDED TO REMOVE THE EXPONENTIAL DECAY IN            */
/*   -PI/3 < ARG(Z) < PI/3 AND THE EXPONENTIAL GROWTH IN                 */
/*   PI/3 < ABS(ARG(Z)) < PI WHERE ZTA=(2/3)*Z*SQRT(Z)                   */

/*   WHILE THE AIRY FUNCTIONS AI(Z) AND DAI(Z)/DZ ARE ANALYTIC IN        */
/*   THE WHOLE Z PLANE, THE CORRESPONDING SCALED FUNCTIONS DEFINED       */
/*   FOR KODE=2 HAVE A CUT ALONG THE NEGATIVE REAL AXIS.                 */
/*   DEFINITIONS AND NOTATION ARE FOUND IN THE NBS HANDBOOK OF           */
/*   MATHEMATICAL FUNCTIONS (REF. 1).                                    */

/*   INPUT                                                               */
/*     Z      - Z=CMPLX(X,Y)                                             */
/*     ID     - ORDER OF DERIVATIVE, ID=0 OR ID=1                        */
/*     KODE   - A PARAMETER TO INDICATE THE SCALING OPTION               */
/*              KODE= 1  RETURNS                                         */
/*                       AI=AI(Z)                ON ID=0 OR              */
/*                       AI=DAI(Z)/DZ            ON ID=1                 */
/*                  = 2  RETURNS                                         */
/*                       AI=EXP(ZTA)*AI(Z)       ON ID=0 OR              */
/*                       AI=EXP(ZTA)*DAI(Z)/DZ   ON ID=1 WHERE           */
/*                       ZTA=(2/3)*Z*SQRT(Z)                             */

/*   OUTPUT                                                              */
/*     AI     - COMPLEX ANSWER DEPENDING ON THE CHOICES FOR ID AND KODE  */
/*     NZ     - UNDERFLOW INDICATOR                                      */
/*              NZ= 0   , NORMAL RETURN                                  */
/*              NZ= 1   , AI=CMPLX(0.0,0.0) DUE TO UNDERFLOW IN          */
/*                        -PI/3 < ARG(Z) < PI/3 ON KODE=1                */
/*     IERR   - ERROR FLAG                                               */
/*              IERR=0, NORMAL RETURN - COMPUTATION COMPLETED            */
/*              IERR=1, INPUT ERROR   - NO COMPUTATION                   */
/*              IERR=2, OVERFLOW      - NO COMPUTATION, REAL(ZTA)        */
/*                      TOO LARGE WITH KODE=1.                           */
/*              IERR=3, ABS(Z) LARGE      - COMPUTATION COMPLETED        */
/*                      LOSSES OF SIGNIFICANCE BY ARGUMENT REDUCTION     */
/*                      PRODUCE LESS THAN HALF OF MACHINE ACCURACY       */
/*              IERR=4, ABS(Z) TOO LARGE  - NO COMPUTATION               */
/*                      COMPLETE LOSS OF ACCURACY BY ARGUMENT REDUCTION  */
/*              IERR=5, ERROR              - NO COMPUTATION,             */
/*                      ALGORITHM TERMINATION CONDITION NOT MET          */


/* ***LONG DESCRIPTION                                                   */

/*   AI AND DAI ARE COMPUTED FOR ABS(Z) > 1.0 FROM THE K BESSEL FUNCTIONS BY */
/*      AI(Z) = C*SQRT(Z)*K(1/3,ZTA) , DAI(Z) = -C*Z*K(2/3,ZTA)          */
/*                     C = 1.0/(PI*SQRT(3.0))                            */
/*                   ZTA = (2/3)*Z**(3/2)                                */

/*   WITH THE POWER SERIES FOR ABS(Z) <= 1.0.                            */

/*   IN MOST COMPLEX VARIABLE COMPUTATION, ONE MUST EVALUATE ELEMENTARY  */
/*   FUNCTIONS.  WHEN THE MAGNITUDE OF Z IS LARGE, LOSSES OF SIGNIFICANCE BY */
/*   ARGUMENT REDUCTION OCCUR.  CONSEQUENTLY, IF THE MAGNITUDE OF ZETA = */
/*   (2/3)*Z**1.5 EXCEEDS U1 = SQRT(0.5/UR), THEN LOSSES EXCEEDING HALF  */
/*   PRECISION ARE LIKELY AND AN ERROR FLAG IERR=3 IS TRIGGERED WHERE UR = */
/*   EPSILON(0.0) = UNIT ROUNDOFF.  ALSO, IF THE MAGNITUDE OF ZETA IS LARGER */
/*   THAN U2=0.5/UR, THEN ALL SIGNIFICANCE IS LOST AND IERR=4.  IN ORDER TO USE */
/*   THE INT FUNCTION, ZETA MUST BE FURTHER RESTRICTED NOT TO EXCEED THE */
/*   LARGEST INTEGER, U3 = HUGE(0).  THUS, THE MAGNITUDE OF ZETA MUST BE */
/*   RESTRICTED BY MIN(U2,U3).  ON 32 BIT MACHINES, U1, U2, AND U3 ARE   */
/*   APPROXIMATELY 2.0E+3, 4.2E+6, 2.1E+9 IN SINGLE PRECISION ARITHMETIC AND */
/*   1.3E+8, 1.8E+16, 2.1E+9 IN DOUBLE PRECISION ARITHMETIC RESPECTIVELY. */
/*   THIS MAKES U2 AND U3 LIMITING IN THEIR RESPECTIVE ARITHMETICS.  THIS MEANS */
/*   THAT THE MAGNITUDE OF Z CANNOT EXCEED 3.1E+4 IN SINGLE AND 2.1E+6 IN */
/*   DOUBLE PRECISION ARITHMETIC.  THIS ALSO MEANS THAT ONE CAN EXPECT TO */
/*   RETAIN, IN THE WORST CASES ON 32 BIT MACHINES, NO DIGITS IN SINGLE  */
/*   PRECISION AND ONLY 7 DIGITS IN DOUBLE PRECISION ARITHMETIC.         */
/*   SIMILAR CONSIDERATIONS HOLD FOR OTHER MACHINES.                     */

/*   THE APPROXIMATE RELATIVE ERROR IN THE MAGNITUDE OF A COMPLEX        */
/*   BESSEL FUNCTION CAN BE EXPRESSED BY P*10**S WHERE P = MAX(UNIT      */
/*   ROUNDOFF,1.0E-18) IS THE NOMINAL PRECISION AND 10**S REPRESENTS     */
/*   THE INCREASE IN ERROR DUE TO ARGUMENT REDUCTION IN THE              */
/*   ELEMENTARY FUNCTIONS.  HERE, S = MAX(1,ABS(LOG10(ABS(Z))),          */
/*   ABS(LOG10(FNU))) APPROXIMATELY (I.E. S=MAX(1,ABS(EXPONENT OF        */
/*   ABS(Z),ABS(EXPONENT OF FNU)) ).  HOWEVER, THE PHASE ANGLE MAY       */
/*   HAVE ONLY ABSOLUTE ACCURACY.  THIS IS MOST LIKELY TO OCCUR WHEN     */
/*   ONE COMPONENT (IN ABSOLUTE VALUE) IS LARGER THAN THE OTHER BY       */
/*   SEVERAL ORDERS OF MAGNITUDE.  IF ONE COMPONENT IS 10**K LARGER      */
/*   THAN THE OTHER, THEN ONE CAN EXPECT ONLY MAX(ABS(LOG10(P))-K,       */
/*   0) SIGNIFICANT DIGITS; OR, STATED ANOTHER WAY, WHEN K EXCEEDS       */
/*   THE EXPONENT OF P, NO SIGNIFICANT DIGITS REMAIN IN THE SMALLER      */
/*   COMPONENT.  HOWEVER, THE PHASE ANGLE RETAINS ABSOLUTE ACCURACY      */
/*   BECAUSE, IN COMPLEX ARITHMETIC WITH PRECISION P, THE SMALLER        */
/*   COMPONENT WILL NOT (AS A RULE) DECREASE BELOW P TIMES THE           */
/*   MAGNITUDE OF THE LARGER COMPONENT.  IN THESE EXTREME CASES,         */
/*   THE PRINCIPAL PHASE ANGLE IS ON THE ORDER OF +P, -P, PI/2-P, OR -PI/2+P. */

/* ***REFERENCES  HANDBOOK OF MATHEMATICAL FUNCTIONS BY M. ABRAMOWITZ AND */
/*      I. A. STEGUN, NBS AMS SERIES 55, U.S. DEPT. OF COMMERCE, 1955.   */

/*    COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT                */
/*      AND LARGE ORDER BY D. E. AMOS, SAND83-0643, MAY 1983             */

/*    A SUBROUTINE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT    */
/*      AND NONNEGATIVE ORDER BY D. E. AMOS, SAND85-1018, MAY 1985       */

/*    A PORTABLE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT      */
/*      AND NONNEGATIVE ORDER BY D. E. AMOS, ACM TRANS. MATH. SOFTWARE,  */
/*      VOL. 12, NO. 3, SEPTEMBER 1986, PP 265-273.                      */

/* ***ROUTINES CALLED  CACAI,CBKNU,I1MACH,R1MACH                         */
/* ***END PROLOGUE  CAIRY                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( id )  FIXED BINARY (31);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( ai )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( ierr )  FIXED BINARY (31);

   DECLARE ( csq, cy(1), s1, s2, trm1, trm2, zta, z3 )  COMPLEX FLOAT (18);
   DECLARE ( aa, ad, ak, alim, atrm, az, az3, bk, ck, dig, dk, d1, d2, elim, 
      fid, fnu, rl, r1m5, sfac, tol, zi, zr, z3i, z3r, bb, alaz )  FLOAT (18);
   DECLARE ( iflag, k, k1, k2, mr, nn )  FIXED BINARY (31);
   DECLARE ( tth  STATIC INITIAL ( 6.66666666666666667E-01)
              , c1  STATIC INITIAL ( 3.55028053887817240E-01)
              , c2  STATIC INITIAL ( 2.58819403792806799E-01)
              , coef  STATIC INITIAL ( 1.83776298473930683E-01)
               )  FLOAT (18);
   DECLARE cone  STATIC INITIAL ( 1+0i )  COMPLEX FLOAT (18);

/****FIRST EXECUTABLE STATEMENT  CAIRY                                  */
   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( iflag, k, k1 );
      PUT SKIP DATA ( k2, mr, nn );
      RESIGNAL;
   END;

   ierr = 0;
   nz = 0;
   IF  id < 0  |   id > 1 THEN
      ierr = 1;
   IF  kode < 1  |   kode > 2 THEN
      ierr = 1;
   IF  ierr ^= 0 THEN
      RETURN;
   az = ABS(z);
   tol = MAX(EPSILON(0.00000000000000000E0), 1.00000000000000000E-18);
   fid = id;
   IF  az <= 1  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     POWER SERIES FOR ABS(Z) <= 1.                                     */
/* ----------------------------------------------------------------------- */
         s1 = cone;
         s2 = cone;
         IF  az < tol THEN  GO TO L30;
         aa = az * az;
         IF  aa >= tol/az  THEN
            DO;
               trm1 = cone;
               trm2 = cone;
               atrm = 1;
               z3 = z * z * z;
               az3 = az * aa;
               ak = 2 + fid;
               bk = 3 - fid - fid;
               ck = 4 - fid;
               dk = 3 + fid + fid;
               d1 = ak * dk;
               d2 = bk * ck;
               ad = MIN(d1,d2);
               ak = 24.0000000000000000E0 + 9.00000000000000000E0 * fid;
               bk = 30.0000000000000000E0 - 9.00000000000000000E0 * fid;
               z3r = REAL(z3);
               z3i = IMAG(z3);
               DO  k = 1 TO  25;
                  trm1 = trm1 * COMPLEX(z3r/d1, z3i/d1 );
                  s1 = s1 + trm1;
                  trm2 = trm2 * COMPLEX(z3r/d2, z3i/d2 );
                  s2 = s2 + trm2;
                  atrm = atrm * az3 / ad;
                  d1 = d1 + ak;
                  d2 = d2 + bk;
                  ad = MIN(d1,d2);
                  IF  atrm < tol*ad THEN
                     LEAVE;
                  ak = ak + 18.0000000000000000E0;
                  bk = bk + 18.0000000000000000E0;
               END;
            END;

         IF  id ^= 1  THEN
            DO;
               ai = s1 * c1 - z * s2 * c2;
               IF  kode  = 1 THEN
                  RETURN;
               zta = z * SQRT(z) * tth;
               ai = ai * EXP(zta);
               RETURN;
            END;
         ai = -s2 * c2;
         IF  az > tol THEN
            ai = ai + z * z * s1 * c1/(1.00000000000000000E0 + fid);
         IF  kode  = 1 THEN
            RETURN;
         zta = z * SQRT(z) * tth;
         ai = ai * EXP(zta);
         RETURN;
      END;
/* ----------------------------------------------------------------------- */
/*     CASE FOR ABS(Z) > 1.0                                             */
/* ----------------------------------------------------------------------- */
   fnu = (1.00000000000000000E0 + fid) / 3.00000000000000000E0;
/* ----------------------------------------------------------------------- */
/*     SET PARAMETERS RELATED TO MACHINE CONSTANTS.                      */
/*     TOL IS THE APPROXIMATE UNIT ROUNDOFF LIMITED TO 1.0E-18.          */
/*     ELIM IS THE APPROXIMATE EXPONENTIAL OVER- AND UNDERFLOW LIMIT.    */
/*     EXP(-ELIM) < EXP(-ALIM)=EXP(-ELIM)/TOL    AND                     */
/*     EXP(ELIM) > EXP(ALIM)=EXP(ELIM)*TOL       ARE INTERVALS NEAR      */
/*     UNDERFLOW AND OVERFLOW LIMITS WHERE SCALED ARITHMETIC IS DONE.    */
/*     RL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC EXPANSION FOR LARGE Z. */
/*     DIG = NUMBER OF BASE 10 DIGITS IN TOL = 10**(-DIG).               */
/* ----------------------------------------------------------------------- */
   k1 = MINEXPONENT(0.00000000000000000E0);
   k2 = MAXEXPONENT(0.00000000000000000E0);
   r1m5 = LOG10( RADIX(0.00000000000000000E0) );
   k = MIN(ABS(k1),ABS(k2));
   elim = 2.30300000000000000E0 * (k*r1m5 - 3.00000000000000000E0);
   k1 = DIGITS(0.00000000000000000E0) - 1;
   aa = r1m5 * k1;
   dig = MIN(aa,18.0000000000000000E0);
   aa = aa * 2.30300000000000000E0;
   alim = elim + MAX(-aa,-41.4500000000000000E0);
   rl = 1.20000000000000000E0 * dig + 3.00000000000000000E0;
   alaz = LOG(az);
/* ----------------------------------------------------------------------- */
/*     TEST FOR RANGE                                                    */
/* ----------------------------------------------------------------------- */
   aa = 0.50000000000000000E0 / tol;
   bb = HUGE(0) * 0.50000000000000000E0;
   aa = MIN(aa,bb);
   aa = aa ** tth;
   IF  az > aa THEN  GO TO L70;
   aa = SQRT(aa);
   IF  az > aa THEN
      ierr = 3;
   csq = SQRT(z);
   zta = z * csq * tth;
/* ----------------------------------------------------------------------- */
/*     RE(ZTA) <= 0 WHEN RE(Z) < 0, ESPECIALLY WHEN IM(Z) IS SMALL       */
/* ----------------------------------------------------------------------- */
   iflag = 0;
   sfac = 1.00000000000000000E0;
   zi = IMAG(z);
   zr = REAL(z);
   ak = IMAG(zta);
   IF  zr < 0  THEN
      DO;
         bk = REAL(zta);
         ck = -ABS(bk);
         zta = COMPLEX(ck, ak);
      END;
   IF  zi  = 0  THEN
      DO;
         IF  zr <= 0  THEN
            DO;
               zta = COMPLEX(0.00000000000000000E0, ak );
            END;
      END;
   aa = REAL(zta);
   IF  aa < 0  |   zr <= 0  THEN
      DO;
         IF  kode ^= 2  THEN
            DO;
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST                                                     */
/* ----------------------------------------------------------------------- */
               IF  aa <= -alim  THEN
                  DO;
                     aa = -aa + 0.25000000000000000E0 * alaz;
                     iflag = 1;
                     sfac = tol;
                           IF  aa > elim THEN  GO TO L50;
                  END;
            END;
/* ----------------------------------------------------------------------- */
/*     CBKNU AND CACAI RETURN EXP(ZTA)*K(FNU,ZTA) ON KODE=2              */
/* ----------------------------------------------------------------------- */
         mr = 1;
         IF  zi < 0 THEN
            mr = -1;
         CALL cacai(zta, fnu, kode, mr, 1, cy, nn, rl, tol, elim, alim);
         IF  nn < 0 THEN  GO TO L60;
         nz = nz + nn;
      END;
   ELSE
      DO;
         IF  kode ^= 2  THEN
            DO;
/* ----------------------------------------------------------------------- */
/*     UNDERFLOW TEST                                                    */
/* ----------------------------------------------------------------------- */
               IF  aa >= alim  THEN
                  DO;
                     aa = -aa - 0.25000000000000000E0 * alaz;
                     iflag = 2;
                     sfac = 1.00000000000000000E0 / tol;
                           IF  aa < -elim THEN  GO TO L40;
                  END;
            END;
         CALL cbknu(zta, fnu, kode, 1, cy, nz, tol, elim, alim);
      END;
   s1 = cy(1) * coef;
   IF  iflag  = 0  THEN
      DO;
         IF  id ^= 1  THEN
            DO;
               ai = csq * s1;
               RETURN;
            END;
         ai = -z * s1;
         RETURN;
      END;
   s1 = s1 * sfac;
   IF  id ^= 1  THEN
      DO;
         s1 = s1 * csq;
         ai = s1 / sfac;
         RETURN;
      END;
   s1 = -s1 * z;
   ai = s1 / sfac;
   RETURN;

L30:
   aa = 1.00000000000000000E+3 * TINY(0.00000000000000000E0);
   s1 = 0;
   IF  id ^= 1  THEN
      DO;
         IF  az > aa THEN
            s1 = c2 * z;
         ai = c1 - s1;
         RETURN;
      END;
   ai = -c2;
   aa = SQRT(aa);
   IF  az > aa THEN
      s1 = z * z * 0.50000000000000000E0;
   ai = ai + s1 * c1;
   RETURN;

L40:
   nz = 1;
   ai = 0;
   RETURN;

L50:
   nz = 0;
   ierr = 2;
   RETURN;

L60:
   IF  nn  = -1 THEN  GO TO L50;
   nz = 0;
   ierr = 5;
   RETURN;

L70:
   ierr = 4;
   nz = 0;
   RETURN;
   END cairy;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbiry: PROCEDURE (z, id, kode, bi, ierr) OPTIONS (REORDER);                     
/* ***BEGIN PROLOGUE  CBIRY                                              */
/* ***DATE WRITTEN   830501   (YYMMDD)                                   */
/* ***REVISION DATE  890801, 930101   (YYMMDD)                           */
/* ***CATEGORY NO.  B5K                                                  */
/* ***KEYWORDS  AIRY FUNCTION,BESSEL FUNCTIONS OF ORDER ONE THIRD        */
/* ***AUTHOR  AMOS, DONALD E., SANDIA NATIONAL LABORATORIES              */
/* ***PURPOSE  TO COMPUTE AIRY FUNCTIONS BI(Z) AND DBI(Z) FOR COMPLEX Z  */
/* ***DESCRIPTION                                                        */

/*   ON KODE=1, CBIRY COMPUTES THE COMPLEX AIRY FUNCTION BI(Z) OR ITS    */
/*   DERIVATIVE DBI(Z)/DZ ON ID=0 OR ID=1 RESPECTIVELY.  ON KODE=2,      */
/*   A SCALING OPTION EXP(-AXZTA)*BI(Z) OR EXP(-AXZTA)*DBI(Z)/DZ         */
/*   IS PROVIDED TO REMOVE THE EXPONENTIAL BEHAVIOR IN BOTH THE LEFT AND */
/*   RIGHT HALF PLANES WHERE ZTA = (2/3)*Z*SQRT(Z) = CMPLX(XZTA,YZTA)    */
/*   AND AXZTA=ABS(XZTA).                                                */
/*   DEFINITIONS AND NOTATION ARE FOUND IN THE NBS HANDBOOK OF MATHEMATICAL */
/*   FUNCTIONS (REF. 1).                                                 */

/*   INPUT                                                               */
/*     Z      - Z=CMPLX(X,Y)                                             */
/*     ID     - ORDER OF DERIVATIVE, ID=0 OR ID=1                        */
/*     KODE   - A PARAMETER TO INDICATE THE SCALING OPTION               */
/*              KODE= 1  RETURNS                                         */
/*                       BI=BI(Z)                 ON ID=0 OR             */
/*                       BI=DBI(Z)/DZ             ON ID=1                */
/*                  = 2  RETURNS                                         */
/*                       BI=EXP(-AXZTA)*BI(Z)     ON ID=0 OR             */
/*                       BI=EXP(-AXZTA)*DBI(Z)/DZ ON ID=1 WHERE          */
/*                       ZTA=(2/3)*Z*SQRT(Z)=CMPLX(XZTA,YZTA)            */
/*                       AND AXZTA=ABS(XZTA)                             */

/*   OUTPUT                                                              */
/*     BI     - COMPLEX ANSWER DEPENDING ON THE CHOICES FOR ID AND KODE  */
/*     IERR   - ERROR FLAG                                               */
/*              IERR=0, NORMAL RETURN - COMPUTATION COMPLETED            */
/*              IERR=1, INPUT ERROR   - NO COMPUTATION                   */
/*              IERR=2, OVERFLOW      - NO COMPUTATION, REAL(Z)          */
/*                      TOO LARGE WITH KODE=1                            */
/*              IERR=3, ABS(Z) LARGE      - COMPUTATION COMPLETED        */
/*                      LOSSES OF SIGNIFICANCE BY ARGUMENT REDUCTION     */
/*                      PRODUCE LESS THAN HALF OF MACHINE ACCURACY       */
/*              IERR=4, ABS(Z) TOO LARGE  - NO COMPUTATION               */
/*                      COMPLETE LOSS OF ACCURACY BY ARGUMENT            */
/*                      REDUCTION                                        */
/*              IERR=5, ERROR              - NO COMPUTATION,             */
/*                      ALGORITHM TERMINATION CONDITION NOT MET          */

/* ***LONG DESCRIPTION                                                   */

/*      BI AND DBI ARE COMPUTED FOR ABS(Z) > 1.0 FROM THE I BESSEL       */
/*      FUNCTIONS BY                                                     */

/*             BI(Z)=C*SQRT(Z)*( I(-1/3,ZTA) + I(1/3,ZTA) )              */
/*            DBI(Z)=C *  Z  * ( I(-2/3,ZTA) + I(2/3,ZTA) )              */
/*                            C=1.0/SQRT(3.0)                            */
/*                            ZTA=(2/3)*Z**(3/2)                         */

/*      WITH THE POWER SERIES FOR ABS(Z) <= 1.0.                         */

/*      IN MOST COMPLEX VARIABLE COMPUTATION, ONE MUST EVALUATE ELE-     */
/*      MENTARY FUNCTIONS. WHEN THE MAGNITUDE OF Z IS LARGE, LOSSES      */
/*      OF SIGNIFICANCE BY ARGUMENT REDUCTION OCCUR. CONSEQUENTLY, IF    */
/*      THE MAGNITUDE OF ZETA=(2/3)*Z**1.5 EXCEEDS U1=SQRT(0.5/UR),      */
/*      THEN LOSSES EXCEEDING HALF PRECISION ARE LIKELY AND AN ERROR     */
/*      FLAG IERR=3 IS TRIGGERED WHERE UR=EPSILON(0.0)=UNIT ROUNDOFF.    */
/*      ALSO, IF THE MAGNITUDE OF ZETA IS LARGER THAN U2=0.5/UR, THEN    */
/*      ALL SIGNIFICANCE IS LOST AND IERR=4. IN ORDER TO USE THE INT     */
/*      FUNCTION, ZETA MUST BE FURTHER RESTRICTED NOT TO EXCEED THE      */
/*      LARGEST INTEGER, U3=HUGE(0). THUS, THE MAGNITUDE OF ZETA         */
/*      MUST BE RESTRICTED BY MIN(U2,U3). ON 32 BIT MACHINES, U1,U2,     */
/*      AND U3 ARE APPROXIMATELY 2.0E+3, 4.2E+6, 2.1E+9 IN SINGLE        */
/*      PRECISION ARITHMETIC AND 1.3E+8, 1.8E+16, 2.1E+9 IN DOUBLE       */
/*      PRECISION ARITHMETIC RESPECTIVELY. THIS MAKES U2 AND U3 LIMIT-   */
/*      ING IN THEIR RESPECTIVE ARITHMETICS. THIS MEANS THAT THE MAG-    */
/*      NITUDE OF Z CANNOT EXCEED 3.1E+4 IN SINGLE AND 2.1E+6 IN         */
/*      DOUBLE PRECISION ARITHMETIC. THIS ALSO MEANS THAT ONE CAN        */
/*      EXPECT TO RETAIN, IN THE WORST CASES ON 32 BIT MACHINES,         */
/*      NO DIGITS IN SINGLE PRECISION AND ONLY 7 DIGITS IN DOUBLE        */
/*      PRECISION ARITHMETIC.                                            */

/*      THE APPROXIMATE RELATIVE ERROR IN THE MAGNITUDE OF A COMPLEX     */
/*      BESSEL FUNCTION CAN BE EXPRESSED BY P*10**S WHERE P=MAX(UNIT     */
/*      ROUNDOFF,1.0E-18) IS THE NOMINAL PRECISION AND 10**S REPRE-      */
/*      SENTS THE INCREASE IN ERROR DUE TO ARGUMENT REDUCTION IN THE     */
/*      ELEMENTARY FUNCTIONS. HERE, S=MAX(1,ABS(LOG10(ABS(Z))),          */
/*      ABS(LOG10(FNU))) APPROXIMATELY (I.E. S=MAX(1,ABS(EXPONENT OF     */
/*      ABS(Z),ABS(EXPONENT OF FNU)) ). HOWEVER, THE PHASE ANGLE MAY     */
/*      HAVE ONLY ABSOLUTE ACCURACY. THIS IS MOST LIKELY TO OCCUR WHEN   */
/*      ONE COMPONENT (IN ABSOLUTE VALUE) IS LARGER THAN THE OTHER BY    */
/*      SEVERAL ORDERS OF MAGNITUDE. IF ONE COMPONENT IS 10**K LARGER    */
/*      THAN THE OTHER, THEN ONE CAN EXPECT ONLY MAX(ABS(LOG10(P))-K,    */
/*      0) SIGNIFICANT DIGITS; OR, STATED ANOTHER WAY, WHEN K EXCEEDS    */
/*      THE EXPONENT OF P, NO SIGNIFICANT DIGITS REMAIN IN THE SMALLER   */
/*      COMPONENT. HOWEVER, THE PHASE ANGLE RETAINS ABSOLUTE ACCURACY    */
/*      BECAUSE, IN COMPLEX ARITHMETIC WITH PRECISION P, THE SMALLER     */
/*      COMPONENT WILL NOT (AS A RULE) DECREASE BELOW P TIMES THE        */
/*      MAGNITUDE OF THE LARGER COMPONENT. IN THESE EXTREME CASES,       */
/*      THE PRINCIPAL PHASE ANGLE IS ON THE ORDER OF +P, -P, PI/2-P,     */
/*      OR -PI/2+P.                                                      */

/* ***REFERENCES  HANDBOOK OF MATHEMATICAL FUNCTIONS BY M. ABRAMOWITZ AND */
/*           I. A. STEGUN, NBS AMS SERIES 55, U.S. DEPT. OF COMMERCE, 1955. */

/*         COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT           */
/*           AND LARGE ORDER BY D. E. AMOS, SAND83-0643, MAY 1983        */

/*         A SUBROUTINE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT */
/*           AND NONNEGATIVE ORDER BY D. E. AMOS, SAND85-1018, MAY 1985  */

/*         A PORTABLE PACKAGE FOR BESSEL FUNCTIONS OF A COMPLEX ARGUMENT */
/*           AND NONNEGATIVE ORDER BY D. E. AMOS, ACM TRANS. MATH. SOFTWARE, */
/*           VOL. 12, NO. 3, SEPTEMBER 1986, PP 265-273.                 */

/* ***ROUTINES CALLED  CBINU,I1MACH,R1MACH                               */
/* ***END PROLOGUE  CBIRY                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( id )  FIXED BINARY (31);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( bi )  COMPLEX FLOAT (18);
   DECLARE ( ierr )  FIXED BINARY (31);

   DECLARE ( csq, cy(2), s1, s2, trm1, trm2, zta, z3 )  COMPLEX FLOAT (18);
   DECLARE ( aa, ad, ak, alim, atrm, az, az3, bb, bk, ck, dig, dk, d1, d2, 
      elim, fid, fmr, fnu, fnul, rl, r1m5, sfac, tol, zi, zr, z3i, z3r )  FLOAT (18);
   DECLARE ( k, k1, k2, nz )  FIXED BINARY (31);
   DECLARE ( tth  STATIC INITIAL ( 6.66666666666666667E-01)
              , c1  STATIC INITIAL ( 6.14926627446000736E-01)
              , c2  STATIC INITIAL ( 4.48288357353826359E-01)
              , coef  STATIC INITIAL ( 5.77350269189625765E-01)
              , pi  STATIC INITIAL ( 3.141592653589793238E0) )  FLOAT (18);
   DECLARE cone  STATIC INITIAL
      ( 1.00000000000000000E0+0.00000000000000000E0i) COMPLEX FLOAT (18);

/* ***FIRST EXECUTABLE STATEMENT  CBIRY                                  */
   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( k, k1, k2, nz );
      RESIGNAL;
   END;

   ierr = 0;
   nz = 0;
   IF  id < 0  |   id > 1 THEN
      ierr = 1;
   IF  kode < 1  |   kode > 2 THEN
      ierr = 1;
   IF  ierr ^= 0 THEN
      RETURN;
   az = ABS(z);
   tol = MAX(EPSILON(0.00000000000000000E0), 1.00000000000000000E-18);
   fid = id;
   IF  az <= 1.00000000000000000E0  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     POWER SERIES FOR ABS(Z) <= 1.                                     */
/* ----------------------------------------------------------------------- */
         s1 = cone;
         s2 = cone;
           IF  az < tol THEN  GO TO L30;
         aa = az * az;
         IF  aa >= tol/az  THEN
            DO;
               trm1 = cone;
               trm2 = cone;
               atrm = 1.00000000000000000E0;
               z3 = z * z * z;
               az3 = az * aa;
               ak = 2.00000000000000000E0 + fid;
               bk = 3.00000000000000000E0 - fid - fid;
               ck = 4.00000000000000000E0 - fid;
               dk = 3.00000000000000000E0 + fid + fid;
               d1 = ak * dk;
               d2 = bk * ck;
               ad = MIN(d1,d2);
               ak = 24.0000000000000000E0 + 9.00000000000000000E0 * fid;
               bk = 30.0000000000000000E0 - 9.00000000000000000E0 * fid;
               z3r = REAL(z3);
               z3i = IMAG(z3);
               DO  k = 1 TO  25;
                  trm1 = trm1 * COMPLEX(z3r/d1, z3i/d1 );
                  s1 = s1 + trm1;
                  trm2 = trm2 * COMPLEX(z3r/d2, z3i/d2 );
                  s2 = s2 + trm2;
                  atrm = atrm * az3 / ad;
                  d1 = d1 + ak;
                  d2 = d2 + bk;
                  ad = MIN(d1,d2);
                  IF  atrm < tol*ad THEN
                     LEAVE;
                  ak = ak + 18.0000000000000000E0;
                  bk = bk + 18.0000000000000000E0;
               END;
            END;

         IF  id ^= 1  THEN
            DO;
               bi = s1 * c1 + z * s2 * c2;
               IF  kode  = 1 THEN
                  RETURN;
               zta = z * SQRT(z) * tth;
               aa = REAL(zta);
               aa = -ABS(aa);
               bi = bi * EXP(aa);
               RETURN;
            END;
         bi = s2 * c2;
         IF  az > tol THEN
            bi = bi + z * z * s1 * c1/(1.00000000000000000E0+fid);
         IF  kode  = 1 THEN
            RETURN;
         zta = z * SQRT(z) * tth;
         aa = REAL(zta);
         aa = -ABS(aa);
         bi = bi * EXP(aa);
         RETURN;
      END;
/* ----------------------------------------------------------------------- */
/*     CASE FOR ABS(Z) > 1.0                                             */
/* ----------------------------------------------------------------------- */
   fnu = (1.00000000000000000E0+fid) / 3.00000000000000000E0;
/* ----------------------------------------------------------------------- */
/*     SET PARAMETERS RELATED TO MACHINE CONSTANTS.                      */
/*     TOL IS THE APPROXIMATE UNIT ROUNDOFF LIMITED TO 1.0E-18.          */
/*     ELIM IS THE APPROXIMATE EXPONENTIAL OVER- AND UNDERFLOW LIMIT.    */
/*     EXP(-ELIM) < EXP(-ALIM)=EXP(-ELIM)/TOL    AND                     */
/*     EXP(ELIM) > EXP(ALIM)=EXP(ELIM)*TOL       ARE INTERVALS NEAR      */
/*     UNDERFLOW AND OVERFLOW LIMITS WHERE SCALED ARITHMETIC IS DONE.    */
/*     RL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC EXPANSION FOR LARGE Z. */
/*     DIG = NUMBER OF BASE 10 DIGITS IN TOL = 10**(-DIG).               */
/*     FNUL IS THE LOWER BOUNDARY OF THE ASYMPTOTIC SERIES FOR LARGE FNU. */
/* ----------------------------------------------------------------------- */
   k1 = MINEXPONENT(0.00000000000000000E0);
   k2 = MAXEXPONENT(0.00000000000000000E0);
   r1m5 = LOG10( RADIX(0.00000000000000000E0) );
   k = MIN(ABS(k1),ABS(k2));
   elim = 2.30300000000000000E0 * (k*r1m5 - 3.00000000000000000E0);
   k1 = DIGITS(0.00000000000000000E0) - 1;
   aa = r1m5 * k1;
   dig = MIN(aa,18.0000000000000000E0);
   aa = aa * 2.30300000000000000E0;
   alim = elim + MAX(-aa,-41.4500000000000000E0);
   rl = 1.20000000000000000E0 * dig + 3.00000000000000000E0;
   fnul = 10.0000000000000000E0 + 6.00000000000000000E0 * (dig - 3.00000000000000000E0); 
       
/* ----------------------------------------------------------------------- */
/*     TEST FOR RANGE                                                    */
/* ----------------------------------------------------------------------- */
   aa = 0.50000000000000000E0 / tol;
   bb = HUGE(0) * 0.50000000000000000E0;
   aa = MIN(aa,bb);
   aa = aa ** tth;
   IF  az > aa THEN  GO TO L60;
   aa = SQRT(aa);
   IF  az > aa THEN
      ierr = 3;
   csq = SQRT(z);
   zta = z * csq * tth;
/* ----------------------------------------------------------------------- */
/*     RE(ZTA) <= 0 WHEN RE(Z) < 0, ESPECIALLY WHEN IM(Z) IS SMALL       */
/* ----------------------------------------------------------------------- */
   sfac = 1.00000000000000000E0;
   zi = IMAG(z);
   zr = REAL(z);
   ak = IMAG(zta);
   IF  zr < 0  THEN
      DO;
         bk = REAL(zta);
         ck = -ABS(bk);
         zta = COMPLEX(ck, ak);
      END;
   IF  zi  = 0  &   zr <= 0 THEN
      zta = COMPLEX(0.00000000000000000E0, ak);
   aa = REAL(zta);
   IF  kode ^= 2  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST                                                     */
/* ----------------------------------------------------------------------- */
         bb = ABS(aa);
         IF  bb >= alim  THEN
            DO;
               bb = bb + 0.25000000000000000E0 * LOG(az);
               sfac = tol;
               IF  bb > elim THEN  GO TO L40;
            END;
      END;
   fmr = 0;
   IF  aa < 0  |   zr <= 0  THEN
      DO;
         fmr = pi;
         IF  zi < 0 THEN
            fmr = -pi;
         zta = -zta;
      END;
/* ----------------------------------------------------------------------- */
/*     AA=FACTOR FOR ANALYTIC CONTINUATION OF I(FNU,ZTA)                 */
/*     KODE=2 RETURNS EXP(-ABS(XZTA))*I(FNU,ZTA) FROM CBINU              */
/* ----------------------------------------------------------------------- */
   CALL cbinu(zta,fnu,kode,1,cy,nz,rl,fnul,tol,elim,alim);
   IF  nz < 0 THEN  GO TO L50;
   aa = fmr * fnu;
   z3 = sfac;
   s1 = cy(1) * COMPLEX(COS(aa), SIN(aa) ) * z3;
   fnu = (2.00000000000000000E0 - fid) / 3.00000000000000000E0;
   CALL cbinu(zta, fnu, kode, 2, cy, nz, rl, fnul, tol, elim, alim);
   cy(1) = cy(1) * z3;
   cy(2) = cy(2) * z3;
/* ----------------------------------------------------------------------- */
/*     BACKWARD RECUR ONE STEP FOR ORDERS -1/3 OR -2/3                   */
/* ----------------------------------------------------------------------- */
   s2 = cy(1) * COMPLEX(fnu+fnu, 0.00000000000000000E0) / zta + cy(2);
   aa = fmr * (fnu-1.00000000000000000E0);
   s1 = (s1 + s2*COMPLEX(COS(aa), SIN(aa) )) * coef;
   IF  id ^= 1  THEN
      DO;
         s1 = csq * s1;
         bi = s1 / sfac;
         RETURN;
      END;
   s1 = z * s1;
   bi = s1 / sfac;
   RETURN;

L30:
   aa = c1 * (1.00000000000000000E0-fid) + fid * c2;
   bi = aa;
   RETURN;

L40:
   nz = 0;
   ierr = 2;
   RETURN;

L50:
   IF  nz  = -1 THEN  GO TO L40;
   nz = 0;
   ierr = 5;
   RETURN;

L60:
   ierr = 4;
   nz = 0;
   RETURN;
   END cbiry;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cunik: PROCEDURE (zr, fnu, ikflg, ipmtr, tol, init, phi, zeta1, zeta2, total,
         cwrk) OPTIONS (REORDER);

/* ***BEGIN PROLOGUE  CUNIK                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*  CUNIK COMPUTES PARAMETERS FOR THE UNIFORM ASYMPTOTIC EXPANSIONS OF   */
/*  THE I AND K FUNCTIONS ON IKFLG= 1 OR 2 RESPECTIVELY BY               */

/*  W(FNU,ZR) = PHI*EXP(ZETA)*SUM                                        */

/*  WHERE     ZETA = -ZETA1 + ZETA2       OR                             */
/*                    ZETA1 - ZETA2                                      */

/*  THE FIRST CALL MUST HAVE INIT=0.  SUBSEQUENT CALLS WITH THE SAME ZR  */
/*  AND FNU WILL RETURN THE I OR K FUNCTION ON IKFLG= 1 OR 2 WITH NO CHANGE */
/*  IN INIT.  CWRK IS A COMPLEX WORK ARRAY.  IPMTR=0 COMPUTES ALL PARAMETERS. */
/*  IPMTR=1 COMPUTES PHI, ZETA1, ZETA2.                                  */

/* ***ROUTINES CALLED  (NONE)                                            */
/* ***END PROLOGUE  CUNIK                                                */

   DECLARE ( zr )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( ikflg )  FIXED BINARY (31);
   DECLARE ( ipmtr )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( init )  FIXED BINARY (31);
   DECLARE ( phi )  COMPLEX FLOAT (18);
   DECLARE ( zeta1 )  COMPLEX FLOAT (18);
   DECLARE ( zeta2 )  COMPLEX FLOAT (18);
   DECLARE ( total )  COMPLEX FLOAT (18);
   DECLARE ( cwrk(16) )  COMPLEX FLOAT (18);

   DECLARE ( cfn, crfn, s, sr, t, t2,  zn )  COMPLEX FLOAT (18);
   DECLARE ( ac, rfn, test, tstr, tsti )  FLOAT (18);
   DECLARE ( i, j, k, l )  FIXED BINARY (31);
   DECLARE ( czero STATIC INITIAL ( 0.00000000000000000E0+0.00000000000000000E0i),
             cone  STATIC INITIAL ( 1.00000000000000000E0+0.00000000000000000E0i))
                COMPLEX FLOAT (18);
   DECLARE ( con(2)  STATIC INITIAL (
                      3.98942280401432678E-01,
                      1.25331413731550025E0 )  )
                FLOAT (18);
   DECLARE ( c(120)  STATIC INITIAL (
                    1.00000000000000000E0,
                   -2.08333333333333333E-01,
                    1.25000000000000000E-01,
                    3.34201388888888889E-01,
                   -4.01041666666666667E-01,
                    7.03125000000000000E-02,
                   -1.02581259645061728E0,
                    1.84646267361111111E0,
                   -8.91210937500000000E-01,
                    7.32421875000000000E-02,
                    4.66958442342624743E0,
                   -1.12070026162229938E+01,
                    8.78912353515625000E0,
                   -2.36408691406250000E0,
                    1.12152099609375000E-01,
                   -2.82120725582002449E+01,
                    8.46362176746007346E+01,
                   -9.18182415432400174E+01,
                    4.25349987453884549E+01,
                   -7.36879435947963170E0,
                    2.27108001708984375E-01,
                    2.12570130039217123E+02,
                   -7.65252468141181642E+02,
                    1.05999045252799988E+03,
                   -6.99579627376132541E+02,
                    2.18190511744211590E+02,
                   -2.64914304869515555E+01,
                    5.72501420974731445E-01,
                   -1.91945766231840700E+03,
                    8.06172218173730938E+03,
                   -1.35865500064341374E+04,
                    1.16553933368645332E+04,
                   -5.30564697861340311E+03,
                    1.20090291321635246E+03,
                   -1.08090919788394656E+02,
                    1.72772750258445740E0,
                    2.02042913309661486E+04,
                   -9.69805983886375135E+04,
                    1.92547001232531532E+05,
                   -2.03400177280415534E+05,
                    1.22200464983017460E+05,
                   -4.11926549688975513E+04,
                    7.10951430248936372E+03,
                   -4.93915304773088012E+02,
                    6.07404200127348304E0,
                   -2.42919187900551333E+05,
                    1.31176361466297720E+06,
                   -2.99801591853810675E+06,
                    3.76327129765640400E+06,
                   -2.81356322658653411E+06,
                    1.26836527332162478E+06,
                   -3.31645172484563578E+05,
                    4.52187689813627263E+04,
                   -2.49983048181120962E+03,
                    2.43805296995560639E+01,
                    3.28446985307203782E+06,
                   -1.97068191184322269E+07,
                    5.09526024926646422E+07,
                   -7.41051482115326577E+07,
                    6.63445122747290267E+07,
                   -3.75671766607633513E+07,
                    1.32887671664218183E+07,
                   -2.78561812808645469E+06,
                    3.08186404612662398E+05,
                   -1.38860897537170405E+04,
                    1.10017140269246738E+02,
                   -4.93292536645099620E+07,
                    3.25573074185765749E+08,
                   -9.39462359681578403E+08,
                    1.55359689957058006E+09,
                   -1.62108055210833708E+09,
                    1.10684281682301447E+09,
                   -4.95889784275030309E+08,
                    1.42062907797533095E+08,
                   -2.44740627257387285E+07,
                    2.24376817792244943E+06,
                   -8.40054336030240853E+04,
                    5.51335896122020586E+02,
                    8.14789096118312115E+08,
                   -5.86648149205184723E+09,
                    1.86882075092958249E+10,
                   -3.46320433881587779E+10,
                    4.12801855797539740E+10,
                   -3.30265997498007231E+10,
                    1.79542137311556001E+10,
                   -6.56329379261928433E+09,
                    1.55927986487925751E+09,
                   -2.25105661889415278E+08,
                    1.73951075539781645E+07,
                   -5.49842327572288687E+05,
                    3.03809051092238427E+03,
                   -1.46792612476956167E+10,
                    1.14498237732025810E+11,
                   -3.99096175224466498E+11,
                    8.19218669548577329E+11,
                   -1.09837515608122331E+12,
                    1.00815810686538209E+12,
                   -6.45364869245376503E+11,
                    2.87900649906150589E+11,
                   -8.78670721780232657E+10,
                    1.76347306068349694E+10,
                   -2.16716498322379509E+09,
                    1.43157876718888981E+08,
                   -3.87183344257261262E+06,
                    1.82577554742931747E+04,
                    2.86464035717679043E+11,
                   -2.40629790002850396E+12,
                    9.10934118523989896E+12,
                   -2.05168994109344374E+13,
                    3.05651255199353206E+13,
                   -3.16670885847851584E+13,
                    2.33483640445818409E+13,
                   -1.23204913055982872E+13,
                    4.61272578084913197E+12,
                   -1.19655288019618160E+12,
                    2.05914503232410016E+11,
                   -2.18229277575292237E+10,
                    1.24700929351271032E+09,
                   -2.91883881222208134E+07,
                    1.18838426256783253E+05 )  )
                FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, j, k, l );
      RESIGNAL;
   END;

   IF  init  = 0  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     INITIALIZE ALL VARIABLES                                          */
/* ----------------------------------------------------------------------- */
         rfn = 1.00000000000000000E0 / fnu;
         crfn = rfn;
         cwrk = czero;
/*     T = ZR*CRFN                                                       */
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST (ZR/FNU TOO SMALL)                                  */
/* ----------------------------------------------------------------------- */
         tstr = REAL(zr);
         tsti = IMAG(zr);
         test = TINY(0.00000000000000000E0) * 1.00000000000000000E+3;
         ac = fnu * test;
         IF  ABS(tstr) <= ac   &   ABS(tsti) <= ac  THEN
            DO;
               ac = 2 * ABS(LOG(test)) + fnu;
               zeta1 = ac;
               zeta2 = fnu;
               phi = cone;
               RETURN;
            END;
         t = zr * crfn;
         s = cone + t * t;
         sr = SQRT(s);
         cfn = fnu;
         zn = (cone+sr) / t;
         zeta1 = cfn * LOG(zn);
         zeta2 = cfn * sr;
         t = cone / sr;
         sr = t * crfn;
         cwrk(16) = SQRT(sr);
         phi = cwrk(16) * con(ikflg);
         IF  ipmtr ^= 0 THEN
            RETURN;
         t2 = cone / s;
         cwrk(1) = cone;
         crfn = cone;
         ac = 1;
         l = 1;
         DO  k = 2 TO  15;
            s = czero;
            DO  j = 1 TO  k;
               l = l + 1;
               s = s * t2 + c(l);
            END;
            crfn = crfn * sr;
            cwrk(k) = crfn * s;
            ac = ac * rfn;
            tstr = REAL(cwrk(k));
            tsti = IMAG(cwrk(k));
            test = ABS(tstr) + ABS(tsti);
            IF  ac < tol   &   test < tol THEN  GO TO L30;
         END;
         k = 15;

L30:
         init = k;
      END;

   IF  ikflg ^= 2  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     COMPUTE SUM FOR THE I FUNCTION                                    */
/* ----------------------------------------------------------------------- */
         total = 0;
         DO i = 1 to init;
            total = total + cwrk(i);
         END;
         phi = cwrk(16) * con(1);
         RETURN;
      END;
/* ----------------------------------------------------------------------- */
/*     COMPUTE SUM FOR THE K FUNCTION                                    */
/* ----------------------------------------------------------------------- */
   s = czero;
   t = cone;
   DO  i = 1 TO  init;
      s = s + t * cwrk(i);
      t = -t;
   END;
   total = s;
   phi = cwrk(16) * con(2);
   RETURN;
   END cunik;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cuoik: PROCEDURE (z, fnu, kode, ikflg, n, y, nuf, tol, elim, alim)
          OPTIONS (REORDER);
/* ***BEGIN PROLOGUE  CUOIK                                              */
/* ***REFER TO  CBESI,CBESK,CBESH                                        */

/*   CUOIK COMPUTES THE LEADING TERMS OF THE UNIFORM ASYMPTOTIC EXPANSIONS */
/*   FOR THE I AND K FUNCTIONS AND COMPARES THEM (IN LOGARITHMIC FORM)   */
/*   TO ALIM AND ELIM FOR OVER AND UNDERFLOW, WHERE ALIM < ELIM.         */
/*   IF THE MAGNITUDE, BASED ON THE LEADING EXPONENTIAL, IS LESS THAN ALIM OR */
/*   GREATER THAN -ALIM, THEN THE RESULT IS ON SCALE.                    */
/*   IF NOT, THEN A REFINED TEST USING OTHER MULTIPLIERS (IN LOGARITHMIC FORM) */
/*   IS MADE BASED ON ELIM.  HERE EXP(-ELIM) = SMALLEST MACHINE NUMBER*1000 */
/*   AND EXP(-ALIM) = EXP(-ELIM)/TOL                                     */

/*   IKFLG=1 MEANS THE I SEQUENCE IS TESTED                              */
/*        =2 MEANS THE K SEQUENCE IS TESTED                              */
/*   NUF = 0 MEANS THE LAST MEMBER OF THE SEQUENCE IS ON SCALE           */
/*       =-1 MEANS AN OVERFLOW WOULD OCCUR                               */
/*   IKFLG=1 AND NUF > 0 MEANS THE LAST NUF Y VALUES WERE SET TO ZERO    */
/*           THE FIRST N-NUF VALUES MUST BE SET BY ANOTHER ROUTINE       */
/*   IKFLG=2 AND NUF = N MEANS ALL Y VALUES WERE SET TO ZERO             */
/*   IKFLG=2 AND 0 < NUF < N NOT CONSIDERED.  Y MUST BE SET BY ANOTHER ROUTINE */

/* ***ROUTINES CALLED  CUCHK,CUNHJ,CUNIK,R1MACH                          */
/* ***END PROLOGUE  CUOIK                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( ikflg )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nuf )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( arg, asum, bsum, cwrk(16), cz, phi, sum, zb, zeta1, zeta2, zn, 
      zr )  COMPLEX FLOAT (18);
   DECLARE ( aarg, aphi, ascle, ax, ay, fnn, gnn, gnu, rcz, x, yy )  FLOAT (18);
   DECLARE ( iform, init, nn, nw )  FIXED BINARY (31);
   DECLARE ( czero STATIC INITIAL ( 0+0i ) )  COMPLEX FLOAT (18);
   DECLARE ( aic   STATIC INITIAL ( 1.265512123484645396E0) )  FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( iform, init, nn, nw );
      RESIGNAL;
   END;

   nuf = 0;
   nn = n;
   x = REAL(z);
   zr = z;
   IF  x < 0 THEN
      zr = -z;
   zb = zr;
   yy = IMAG(zr);
   ax = ABS(x) * 1.73205080756887000E0;
   ay = ABS(yy);
   iform = 1;
   IF  ay > ax THEN
      iform = 2;
   gnu = MAX(fnu, 1.00000000000000000E0);
   IF  ikflg ^= 1  THEN
      DO;
         fnn = nn;
         gnn = fnu + fnn - 1.00000000000000000E0;
         gnu = MAX(gnn, fnn);
      END;
/* ----------------------------------------------------------------------- */
/*     ONLY THE MAGNITUDE OF ARG AND PHI ARE NEEDED ALONG WITH THE       */
/*     REAL PARTS OF ZETA1, ZETA2 AND ZB.  NO ATTEMPT IS MADE TO GET     */
/*     THE SIGN OF THE IMAGINARY PART CORRECT.                           */
/* ----------------------------------------------------------------------- */
   IF  iform ^= 2  THEN
      DO;
         init = 0;
         CALL cunik(zr, gnu, ikflg, 1, tol, init, phi, zeta1, zeta2, sum, cwrk);
         cz = -zeta1 + zeta2;
      END;
   ELSE
      DO;
         zn = -zr * 1.00000000000000000E0i ;
         IF  yy <= 0  THEN
            DO;
               zn = CONJG(-zn);
            END;
         CALL cunhj(zn, gnu, 1, tol, phi, arg, zeta1, zeta2, asum, bsum);
         cz = -zeta1 + zeta2;
         aarg = ABS(arg);
      END;
   IF  kode  = 2 THEN
      cz = cz - zb;
   IF  ikflg  = 2 THEN
      cz = -cz;
   aphi = ABS(phi);
   rcz = REAL(cz);
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST                                                     */
/* ----------------------------------------------------------------------- */
   IF  rcz <= elim  THEN
      DO;
         IF  rcz >= alim  THEN
            DO;
               rcz = rcz + LOG(aphi);
               IF  iform  = 2 THEN
                  rcz = rcz - 0.25000000000000000E0 * LOG(aarg) - aic;
               IF  rcz > elim THEN  GO TO L80;
            END;
         ELSE
            DO;
/* ----------------------------------------------------------------------- */
/*     UNDERFLOW TEST                                                    */
/* ----------------------------------------------------------------------- */
               IF  rcz >= -elim  THEN
                  DO;
                     IF  rcz > -alim THEN  GO TO L40;
                     rcz = rcz + LOG(aphi);
                     IF  iform  = 2 THEN
                        rcz = rcz - 0.25000000000000000E0 * LOG(aarg) - aic;
                     IF  rcz > -elim THEN  GO TO L30;
                  END;

L10:
               y(*) = czero; /* y(1:nn) = czero */
               nuf = nn;
               RETURN;

L30:
               ascle = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
               cz = cz + LOG(phi);
               IF  iform ^= 1  THEN
                  DO;
                     cz = cz - 0.25000000000000000E0 * LOG(arg) - aic;
                  END;
               ax = EXP(rcz) / tol;
               ay =  IMAG(cz);
               cz = ax * COMPLEX(COS(ay), SIN(ay));
               CALL cuchk(cz, nw, ascle, tol);
               IF  nw  = 1 THEN  GO TO L10;
            END;

L40:
         IF  ikflg  = 2 THEN
            RETURN;
         IF  n  = 1 THEN
            RETURN;
/* ----------------------------------------------------------------------- */
/*     SET UNDERFLOWS ON I SEQUENCE                                      */
/* ----------------------------------------------------------------------- */
L50:
         gnu = fnu + (nn-1);
         IF  iform ^= 2  THEN
            DO;
               init = 0;
               CALL cunik(zr, gnu, ikflg, 1, tol, init, phi, zeta1, zeta2, sum, cwrk);
               cz = -zeta1 + zeta2;
            END;
         ELSE
            DO;
               CALL cunhj(zn, gnu, 1, tol, phi, arg, zeta1, zeta2, asum, bsum);
               cz = -zeta1 + zeta2;
               aarg = ABS(arg);
            END;
         IF  kode  = 2 THEN
            cz = cz - zb;
         aphi = ABS(phi);
         rcz = REAL(cz);
         IF  rcz >= -elim  THEN
            DO;
               IF  rcz > -alim THEN
                  RETURN;
               rcz = rcz + LOG(aphi);
               IF  iform  = 2 THEN
                  rcz = rcz - 0.25000000000000000E0 * LOG(aarg) - aic;
               IF  rcz > -elim THEN  GO TO L70;
            END;

L60:
         y(nn) = czero;
         nn = nn - 1;
         nuf = nuf + 1;
         IF  nn  = 0 THEN
            RETURN;
         GO TO L50;

L70:
         ascle = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
         cz = cz + LOG(phi);
         IF  iform ^= 1  THEN
            DO;
               cz = cz - 0.25000000000000000E0 * LOG(arg) - aic;
            END;
         ax = EXP(rcz) / tol;
         ay = IMAG(cz);
         cz = ax * COMPLEX(COS(ay), SIN(ay) );
         CALL cuchk(cz, nw, ascle, tol);
         IF  nw  = 1 THEN  GO TO L60;
         RETURN;
      END;

L80:
   nuf = -1;
   RETURN;
   END cuoik;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cwrsk: PROCEDURE (zr, fnu, kode, n, y, nz, cw, tol, elim, alim)
         OPTIONS (REORDER);

/* ***BEGIN PROLOGUE  CWRSK                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*     CWRSK COMPUTES THE I BESSEL FUNCTION FOR RE(Z) >= 0.0 BY          */
/*     NORMALIZING THE I FUNCTION RATIOS FROM CRATI BY THE WRONSKIAN     */

/* ***ROUTINES CALLED  CBKNU,CRATI,R1MACH                                */
/* ***END PROLOGUE  CWRSK                                                */

   DECLARE ( zr )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( cw(2) )  COMPLEX FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( cinu, cscl, ct, c1, c2, rct, st )  COMPLEX FLOAT (18);
   DECLARE ( act, acw, ascle, s1, s2, yy )  FLOAT (18);
   DECLARE ( i, nw )  FIXED BINARY (31);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, nw );
      RESIGNAL;
   END;

/* ----------------------------------------------------------------------- */
/*     I(FNU+I-1,Z) BY BACKWARD RECURRENCE FOR RATIOS                    */
/*     Y(I)=I(FNU+I,Z)/I(FNU+I-1,Z) FROM CRATI NORMALIZED BY THE         */
/*     WRONSKIAN WITH K(FNU,Z) AND K(FNU+1,Z) FROM CBKNU.                */
/* ----------------------------------------------------------------------- */
   nz = 0;
   CALL cbknu(zr, fnu, kode, 2, cw, nw, tol, elim, alim);
   IF  nw  = 0  THEN
      DO;
         CALL crati(zr, fnu, n, y, tol);
/* ----------------------------------------------------------------------- */
/*     RECUR FORWARD ON I(FNU+1,Z) = R(FNU,Z)*I(FNU,Z),                  */
/*     R(FNU+J-1,Z)=Y(J),  J=1,...,N                                     */
/* ----------------------------------------------------------------------- */
         cinu = 1+0I; 
         IF  kode ^= 1  THEN
            DO;
               yy = IMAG(zr);
               s1 = COS(yy);
               s2 = SIN(yy);
               cinu = COMPLEX(s1, s2);
            END;
/* ----------------------------------------------------------------------- */
/*     ON LOW EXPONENT MACHINES THE K FUNCTIONS CAN BE CLOSE TO BOTH THE */
/*     UNDER AND OVERFLOW LIMITS AND THE NORMALIZATION MUST BE SCALED TO */
/*     PREVENT OVER OR UNDERFLOW.  CUOIK HAS DETERMINED THAT THE RESULT  */
/*     IS ON SCALE.                                                      */
/* ----------------------------------------------------------------------- */
         acw = ABS(cw(2));
         ascle = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
         cscl = 1.00000000000000000E0;
         IF  acw <= ascle  THEN
            DO;
               cscl = COMPLEX(1.00000000000000000E0/tol, 0);
            END;
         ELSE
            DO;
               ascle = 1.00000000000000000E0 / ascle;
               IF  acw >= ascle  THEN
                  DO;
                     cscl = COMPLEX(tol, 0);
                  END;
            END;
         c1 = cw(1) * cscl;
         c2 = cw(2) * cscl;
         st = y(1);
/* ----------------------------------------------------------------------- */
/*     CINU=CINU*(CONJG(CT)/ABS(CT))*(1.0/ABS(CT) PREVENTS               */
/*     UNDER- OR OVERFLOW PREMATURELY BY SQUARING ABS(CT)                */
/* ----------------------------------------------------------------------- */
         ct = zr * (c2 + st*c1);
         act = ABS(ct);
         rct = COMPLEX(1.00000000000000000E0/act, 0);
         ct = CONJG(ct) * rct;
         cinu = cinu * rct * ct;
         y(1) = cinu * cscl;
         IF  n  = 1 THEN
            RETURN;
         DO  i = 2 TO  n;
            cinu = st * cinu;
            st = y(i);
            y(i) = cinu * cscl;
         END;
         RETURN;
      END;
   nz = -1;
   IF  nw  = -2 THEN
      nz = -2;
   RETURN;
   END cwrsk;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cmlri: PROCEDURE (z, fnu, kode, n, y, nz, tol) OPTIONS (REORDER);               
/* ***BEGIN PROLOGUE  CMLRI                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*     CMLRI COMPUTES THE I BESSEL FUNCTION FOR RE(Z) >= 0.0 BY THE      */
/*     MILLER ALGORITHM NORMALIZED BY A NEUMANN SERIES.                  */

/* ***ROUTINES CALLED  GAMLN,R1MACH                                      */
/* ***END PROLOGUE  CMLRI                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);

   DECLARE ( ck, cnorm, pt, p1, p2, rz, sum )  COMPLEX FLOAT (18);
   DECLARE ( ack, ak, ap, at, az, bk, fkap, fkk, flam, fnf, rho, rho2, scle, 
      tfnf, tst, x )  FLOAT (18);
   DECLARE ( i, iaz, ifnu, inu, itime, k, kk, km, m )  FIXED BINARY (31);

   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i),
             ctwo   STATIC INITIAL (
               2.00000000000000000E0+0.00000000000000000E0i) ) COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, iaz, ifnu );
      PUT SKIP DATA ( inu, itime, k );
      PUT SKIP DATA ( kk, km, m );
      RESIGNAL;
   END;

   scle = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
   nz = 0;
   az = ABS(z);
   x = REAL(z);
   iaz = az;
   ifnu = fnu;
   inu = ifnu + n - 1;
   at = iaz + 1;
   ck = COMPLEX(at, 0) / z;
   rz = ctwo / z;
   p1 = czero;
   p2 = cone;
   ack = (at + 1.00000000000000000E0) / az;
   rho = ack + SQRT(ack*ack - 1.00000000000000000E0);
   rho2 = rho * rho;
   tst = (rho2+rho2) / ((rho2-1.00000000000000000E0)*(rho-1.00000000000000000E0));
   tst = tst / tol;
/* ----------------------------------------------------------------------- */
/*     COMPUTE RELATIVE TRUNCATION ERROR INDEX FOR SERIES                */
/* ----------------------------------------------------------------------- */
   ak = at;
   DO  i = 1 TO  80;
      pt = p2;
      p2 = p1 - ck * p2;
      p1 = pt;
      ck = ck + rz;
      ap = ABS(p2);
      IF  ap > tst*ak*ak THEN  GO TO L20;
      ak = ak + 1.00000000000000000E0;
   END;
   GO TO L90;

L20:
   i = i + 1;
   k = 0;
   IF  inu >= iaz  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     COMPUTE RELATIVE TRUNCATION ERROR FOR RATIOS                      */
/* ----------------------------------------------------------------------- */
         p1 = czero;
         p2 = cone;
         at = inu + 1;
         ck = COMPLEX(at, 0) / z;
         ack = at / az;
         tst = SQRT(ack/tol);
         itime = 1;
         DO  k = 1 TO  80;
            pt = p2;
            p2 = p1 - ck * p2;
            p1 = pt;
            ck = ck + rz;
            ap = ABS(p2);
            IF  ap >= tst  THEN
               DO;
                  IF  itime  = 2 THEN  GO TO L40;
                  ack = ABS(ck);
                  flam = ack + SQRT(ack*ack - 1.00000000000000000E0);
                  fkap = ap / ABS(p1);
                  rho = MIN(flam,fkap);
                  tst = tst * SQRT(rho/(rho*rho - 1.00000000000000000E0));
                  itime = 2;
               END;
         END;
         GO TO L90;
      END;
/* ----------------------------------------------------------------------- */
/*     BACKWARD RECURRENCE AND SUM NORMALIZING RELATION                  */
/* ----------------------------------------------------------------------- */
L40:
   k = k + 1;
   kk = MAX(i+iaz, k+inu);
   fkk = kk;
   p1 = czero;
/* ----------------------------------------------------------------------- */
/*     SCALE P2 AND SUM BY SCLE                                          */
/* ----------------------------------------------------------------------- */
   p2 = COMPLEX(scle, 0);
   fnf = fnu - ifnu;
   tfnf = fnf + fnf;
   bk = gamln(fkk+tfnf+1.00000000000000000E0) - gamln(fkk+1.00000000000000000E0) 
      - gamln(tfnf+1.00000000000000000E0);
   bk = EXP(bk);
   sum = czero;
   km = kk - inu;
   DO  i = 1 TO  km;
      pt = p2;
      p2 = p1 + COMPLEX(fkk+fnf, 0) * rz * p2;
      p1 = pt;
      ak = 1.00000000000000000E0 - tfnf / (fkk+tfnf);
      ack = bk * ak;
      sum = sum + COMPLEX(ack+bk, 0) * p1;
      bk = ack;
      fkk = fkk - 1.00000000000000000E0;
   END;
   y(n) = p2;
   IF  n ^= 1  THEN
      DO;
         DO  i = 2 TO  n;
            pt = p2;
            p2 = p1 + COMPLEX(fkk+fnf, 0) * rz * p2;
            p1 = pt;
            ak = 1.00000000000000000E0 - tfnf / (fkk+tfnf);
            ack = bk * ak;
            sum = sum + COMPLEX(ack+bk, 0) * p1;
            bk = ack;
            fkk = fkk - 1.00000000000000000E0;
            m = n - i + 1;
            y(m) = p2;
         END;
      END;
   IF  ifnu > 0  THEN
      DO;
         DO  i = 1 TO  ifnu;
            pt = p2;
            p2 = p1 + COMPLEX(fkk+fnf, 0) * rz * p2;
            p1 = pt;
            ak = 1.00000000000000000E0 - tfnf / (fkk+tfnf);
            ack = bk * ak;
            sum = sum + COMPLEX(ack+bk, 0) * p1;
            bk = ack;
            fkk = fkk - 1.00000000000000000E0;
         END;
      END;
   pt = z;
   IF  kode  = 2 THEN
      pt = pt - x;
   p1 = -fnf * LOG(rz) + pt;
   ap = gamln(1.00000000000000000E0+fnf);
   pt = p1 - ap;
/* ----------------------------------------------------------------------- */
/*     THE DIVISION EXP(PT)/(SUM+P2) IS ALTERED TO AVOID OVERFLOW        */
/*     IN THE DENOMINATOR BY SQUARING LARGE QUANTITIES                   */
/* ----------------------------------------------------------------------- */
   p2 = p2 + sum;
   ap = ABS(p2);
   p1 = COMPLEX(1.00000000000000000E0/ap, 0);
   ck = EXP(pt) * p1;
   pt = CONJG(p2) * p1;
   cnorm = ck * pt;
   y(*) = y(*) * cnorm;
   RETURN;

L90:
   nz = -2;
   RETURN;
   END cmlri;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cunhj: PROCEDURE (z, fnu, ipmtr, tol, phi, arg, zeta1, zeta2, asum, bsum)
          OPTIONS (REORDER);
/* ***BEGIN PROLOGUE  CUNHJ                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*  REFERENCES                                                           */
/*      HANDBOOK OF MATHEMATICAL FUNCTIONS BY M. ABRAMOWITZ AND I.A.     */
/*      STEGUN, AMS55, NATIONAL BUREAU OF STANDARDS, 1965, CHAPTER 9.    */

/*      ASYMPTOTICS AND SPECIAL FUNCTIONS BY F.W.J. OLVER, ACADEMIC      */
/*      PRESS, N.Y., 1974, PAGE 420                                      */

/*  ABSTRACT                                                             */
/*      CUNHJ COMPUTES PARAMETERS FOR BESSEL FUNCTIONS C(FNU,Z) =        */
/*      J(FNU,Z), Y(FNU,Z) OR H(I,FNU,Z) I=1,2 FOR LARGE ORDERS FNU      */
/*      BY MEANS OF THE UNIFORM ASYMPTOTIC EXPANSION                     */

/*      C(FNU,Z) = C1*PHI*( ASUM*AIRY(ARG) + C2*BSUM*DAIRY(ARG) )        */

/*      FOR PROPER CHOICES OF C1, C2, AIRY AND DAIRY WHERE AIRY IS       */
/*      AN AIRY FUNCTION AND DAIRY IS ITS DERIVATIVE.                    */

/*            (2/3)*FNU*ZETA**1.5 = ZETA1-ZETA2,                         */

/*      ZETA1 = 0.5*FNU*LOG((1+W)/(1-W)), ZETA2 = FNU*W FOR SCALING      */
/*      PURPOSES IN AIRY FUNCTIONS FROM CAIRY OR CBIRY.                  */

/*      MCONJ = SIGN OF AIMAG(Z), BUT IS AMBIGUOUS WHEN Z IS REAL AND    */
/*      MUST BE SPECIFIED.  IPMTR=0 RETURNS ALL PARAMETERS.  IPMTR =     */
/*      1 COMPUTES ALL EXCEPT ASUM AND BSUM.                             */

/* ***ROUTINES CALLED  (NONE)                                            */
/* ***END PROLOGUE  CUNHJ                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( ipmtr )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( phi )  COMPLEX FLOAT (18);
   DECLARE ( arg )  COMPLEX FLOAT (18);
   DECLARE ( zeta1 )  COMPLEX FLOAT (18);
   DECLARE ( zeta2 )  COMPLEX FLOAT (18);
   DECLARE ( asum )  COMPLEX FLOAT (18);
   DECLARE ( bsum )  COMPLEX FLOAT (18);

   DECLARE ( cfnu, cr(14), dr(14), p(30), przth, ptfn, rtzta, rzth, suma, 
      sumb, tfn, t2, up(14), w, w2, za, zb, zc, zeta, zth )  COMPLEX FLOAT (18);
   DECLARE ( ang, ap(30), atol, aw2, azth, btol, fn13, fn23, pp, rfn13, rfnu, 
      rfnu2, wi, wr, zci, zcr, zetai, zetar, zthi, zthr, asumr, asumi, bsumr, 
      bsumi, test, tstr, tsti, ac )  FLOAT (18);
   DECLARE ( ias, ibs, is, j, jr, ju, k, kmax, kp1, ks, l, lr, lrp1, l1, l2, 
      m )  FIXED BINARY (31);
   DECLARE ( ar(14)  STATIC INITIAL (
                    1.00000000000000000E0,
                    1.04166666666666667E-01,
                    8.35503472222222222E-02,
                    1.28226574556327160E-01,
                    2.91849026464140464E-01,
                    8.81627267443757652E-01,
                    3.32140828186276754E0,
                    1.49957629868625547E+01,
                    7.89230130115865181E+01,
                    4.74451538868264323E+02,
                    3.20749009089066193E+03,
                    2.40865496408740049E+04,
                    1.98923119169509794E+05,
                    1.79190200777534383E+06 )  )
                FLOAT (18);
   DECLARE ( br(14)  STATIC INITIAL (
                    1.00000000000000000E0,
                   -1.45833333333333333E-01,
                   -9.87413194444444444E-02,
                   -1.43312053915895062E-01,
                   -3.17227202678413548E-01,
                   -9.42429147957120249E-01,
                   -3.51120304082635426E0,
                   -1.57272636203680451E+01,
                   -8.22814390971859444E+01,
                   -4.92355370523670524E+02,
                   -3.31621856854797251E+03,
                   -2.48276742452085896E+04,
                   -2.04526587315129788E+05,
                   -1.83844491706820990E+06 )  )
                FLOAT (18);
   DECLARE ( c(105)  STATIC INITIAL (
                    1.00000000000000000E0,
                   -2.08333333333333333E-01,
                    1.25000000000000000E-01,
                    3.34201388888888889E-01,
                   -4.01041666666666667E-01,
                    7.03125000000000000E-02,
                   -1.02581259645061728E0,
                    1.84646267361111111E0,
                   -8.91210937500000000E-01,
                    7.32421875000000000E-02,
                    4.66958442342624743E0,
                   -1.12070026162229938E+01,
                    8.78912353515625000E0,
                   -2.36408691406250000E0,
                    1.12152099609375000E-01,
                   -2.82120725582002449E+01,
                    8.46362176746007346E+01,
                   -9.18182415432400174E+01,
                    4.25349987453884549E+01,
                   -7.36879435947963170E0,
                    2.27108001708984375E-01,
                    2.12570130039217123E+02,
                   -7.65252468141181642E+02,
                    1.05999045252799988E+03,
                   -6.99579627376132541E+02,
                    2.18190511744211590E+02,
                   -2.64914304869515555E+01,
                    5.72501420974731445E-01,
                   -1.91945766231840700E+03,
                    8.06172218173730938E+03,
                   -1.35865500064341374E+04,
                    1.16553933368645332E+04,
                   -5.30564697861340311E+03,
                    1.20090291321635246E+03,
                   -1.08090919788394656E+02,
                    1.72772750258445740E0,
                    2.02042913309661486E+04,
                   -9.69805983886375135E+04,
                    1.92547001232531532E+05,
                   -2.03400177280415534E+05,
                    1.22200464983017460E+05,
                   -4.11926549688975513E+04,
                    7.10951430248936372E+03,
                   -4.93915304773088012E+02,
                    6.07404200127348304E0,
                   -2.42919187900551333E+05,
                    1.31176361466297720E+06,
                   -2.99801591853810675E+06,
                    3.76327129765640400E+06,
                   -2.81356322658653411E+06,
                    1.26836527332162478E+06,
                   -3.31645172484563578E+05,
                    4.52187689813627263E+04,
                   -2.49983048181120962E+03,
                    2.43805296995560639E+01,
                    3.28446985307203782E+06,
                   -1.97068191184322269E+07,
                    5.09526024926646422E+07,
                   -7.41051482115326577E+07,
                    6.63445122747290267E+07,
                   -3.75671766607633513E+07,
                    1.32887671664218183E+07,
                   -2.78561812808645469E+06,
                    3.08186404612662398E+05,
                   -1.38860897537170405E+04,
                    1.10017140269246738E+02,
                   -4.93292536645099620E+07,
                    3.25573074185765749E+08,
                   -9.39462359681578403E+08,
                    1.55359689957058006E+09,
                   -1.62108055210833708E+09,
                    1.10684281682301447E+09,
                   -4.95889784275030309E+08,
                    1.42062907797533095E+08,
                   -2.44740627257387285E+07,
                    2.24376817792244943E+06,
                   -8.40054336030240853E+04,
                    5.51335896122020586E+02,
                    8.14789096118312115E+08,
                   -5.86648149205184723E+09,
                    1.86882075092958249E+10,
                   -3.46320433881587779E+10,
                    4.12801855797539740E+10,
                   -3.30265997498007231E+10,
                    1.79542137311556001E+10,
                   -6.56329379261928433E+09,
                    1.55927986487925751E+09,
                   -2.25105661889415278E+08,
                    1.73951075539781645E+07,
                   -5.49842327572288687E+05,
                    3.03809051092238427E+03,
                   -1.46792612476956167E+10,
                    1.14498237732025810E+11,
                   -3.99096175224466498E+11,
                    8.19218669548577329E+11,
                   -1.09837515608122331E+12,
                    1.00815810686538209E+12,
                   -6.45364869245376503E+11,
                    2.87900649906150589E+11,
                   -8.78670721780232657E+10,
                    1.76347306068349694E+10,
                   -2.16716498322379509E+09,
                    1.43157876718888981E+08,
                   -3.87183344257261262E+06,
                    1.82577554742931747E+04 )  )
                FLOAT (18);
   DECLARE ( alfa1(30)  STATIC INITIAL (
                   -4.44444444444444444E-03,
                   -9.22077922077922078E-04,
                   -8.84892884892884893E-05,
                    1.65927687832449737E-04,
                    2.46691372741792910E-04,
                    2.65995589346254780E-04,
                    2.61824297061500945E-04,
                    2.48730437344655609E-04,
                    2.32721040083232098E-04,
                    2.16362485712365082E-04,
                    2.00738858762752355E-04,
                    1.86267636637545172E-04,
                    1.73060775917876493E-04,
                    1.61091705929015752E-04,
                    1.50274774160908134E-04,
                    1.40503497391269794E-04,
                    1.31668816545922806E-04,
                    1.23667445598253261E-04,
                    1.16405271474737902E-04,
                    1.09798298372713369E-04,
                    1.03772410422992823E-04,
                    9.82626078369363448E-05,
                    9.32120517249503256E-05,
                    8.85710852478711718E-05,
                    8.42963105715700223E-05,
                    8.03497548407791151E-05,
                    7.66981345359207388E-05,
                    7.33122157481777809E-05,
                    7.01662625163141333E-05,
                    6.72375633790160292E-05 )  )
                FLOAT (18);
   DECLARE ( alfa2(30)  STATIC INITIAL (
                    6.93735541354588974E-04,
                    2.32241745182921654E-04,
                   -1.41986273556691197E-05,
                   -1.16444931672048640E-04,
                   -1.50803558053048762E-04,
                   -1.55121924918096223E-04,
                   -1.46809756646465549E-04,
                   -1.33815503867491367E-04,
                   -1.19744975684254051E-04,
                   -1.06184319207974020E-04,
                   -9.37699549891194492E-05,
                   -8.26923045588193274E-05,
                   -7.29374348155221211E-05,
                   -6.44042357721016283E-05,
                   -5.69611566009369048E-05,
                   -5.04731044303561628E-05,
                   -4.48134868008882786E-05,
                   -3.98688727717598864E-05,
                   -3.55400532972042498E-05,
                   -3.17414256609022480E-05,
                   -2.83996793904174811E-05,
                   -2.54522720634870566E-05,
                   -2.28459297164724555E-05,
                   -2.05352753106480604E-05,
                   -1.84816217627666085E-05,
                   -1.66519330021393806E-05,
                   -1.50179412980119482E-05,
                   -1.35554031379040526E-05,
                   -1.22434746473858131E-05,
                   -1.10641884811308169E-05 )  )
                FLOAT (18);
   DECLARE ( alfa3(30)  STATIC INITIAL (
                   -3.54211971457743841E-04,
                   -1.56161263945159416E-04,
                    3.04465503594936410E-05,
                    1.30198655773242693E-04,
                    1.67471106699712269E-04,
                    1.70222587683592569E-04,
                    1.56501427608594704E-04,
                    1.36339170977445120E-04,
                    1.14886692029825128E-04,
                    9.45869093034688111E-05,
                    7.64498419250898258E-05,
                    6.07570334965197354E-05,
                    4.74394299290508799E-05,
                    3.62757512005344297E-05,
                    2.69939714979224901E-05,
                    1.93210938247939253E-05,
                    1.30056674793963203E-05,
                    7.82620866744496661E-06,
                    3.59257485819351583E-06,
                    1.44040049814251817E-07,
                   -2.65396769697939116E-06,
                   -4.91346867098485910E-06,
                   -6.72739296091248287E-06,
                   -8.17269379678657923E-06,
                   -9.31304715093561232E-06,
                   -1.02011418798016441E-05,
                   -1.08805962510592880E-05,
                   -1.13875481509603555E-05,
                   -1.17519675674556414E-05,
                   -1.19987364870944141E-05 )  )
                FLOAT (18);
   DECLARE ( alfa4(30)  STATIC INITIAL (
                    3.78194199201772914E-04,
                    2.02471952761816167E-04,
                   -6.37938506318862408E-05,
                   -2.38598230603005903E-04,
                   -3.10916256027361568E-04,
                   -3.13680115247576316E-04,
                   -2.78950273791323387E-04,
                   -2.28564082619141374E-04,
                   -1.75245280340846749E-04,
                   -1.25544063060690348E-04,
                   -8.22982872820208365E-05,
                   -4.62860730588116458E-05,
                   -1.72334302366962267E-05,
                    5.60690482304602267E-06,
                    2.31395443148286800E-05,
                    3.62642745856793957E-05,
                    4.58006124490188752E-05,
                    5.24595294959114050E-05,
                    5.68396208545815266E-05,
                    5.94349820393104052E-05,
                    6.06478527578421742E-05,
                    6.08023907788436497E-05,
                    6.01577894539460388E-05,
                    5.89199657344698500E-05,
                    5.72515823777593053E-05,
                    5.52804375585852577E-05,
                    5.31063773802880170E-05,
                    5.08069302012325706E-05,
                    4.84418647620094842E-05,
                    4.60568581607475370E-05 )  )
                FLOAT (18);
   DECLARE ( alfa5(30)  STATIC INITIAL (
                   -6.91141397288294174E-04,
                   -4.29976633058871912E-04,
                    1.83067735980039018E-04,
                    6.60088147542014144E-04,
                    8.75964969951185931E-04,
                    8.77335235958235514E-04,
                    7.49369585378990637E-04,
                    5.63832329756980918E-04,
                    3.68059319971443156E-04,
                    1.88464535514455599E-04,
                    3.70663057664904149E-05,
                   -8.28520220232137023E-05,
                   -1.72751952869172998E-04,
                   -2.36314873605872983E-04,
                   -2.77966150694906658E-04,
                   -3.02079514155456919E-04,
                   -3.12594712643820127E-04,
                   -3.12872558758067163E-04,
                   -3.05678038466324377E-04,
                   -2.93226470614557331E-04,
                   -2.77255655582934777E-04,
                   -2.59103928467031709E-04,
                   -2.39784014396480342E-04,
                   -2.20048260045422848E-04,
                   -2.00443911094971498E-04,
                   -1.81358692210970687E-04,
                   -1.63057674478657464E-04,
                   -1.45712672175205844E-04,
                   -1.29425421983924587E-04,
                   -1.14245691942445952E-04 )  )
                FLOAT (18);
   DECLARE ( alfa6(30)  STATIC INITIAL (
                    1.92821964248775885E-03,
                    1.35592576302022234E-03,
                   -7.17858090421302995E-04,
                   -2.58084802575270346E-03,
                   -3.49271130826168475E-03,
                   -3.46986299340960628E-03,
                   -2.82285233351310182E-03,
                   -1.88103076404891354E-03,
                   -8.89531718383947600E-04,
                    3.87912102631035228E-06,
                    7.28688540119691412E-04,
                    1.26566373053457758E-03,
                    1.62518158372674427E-03,
                    1.83203153216373172E-03,
                    1.91588388990527909E-03,
                    1.90588846755546138E-03,
                    1.82798982421825727E-03,
                    1.70389506421121530E-03,
                    1.55097127171097686E-03,
                    1.38261421852276159E-03,
                    1.20881424230064774E-03,
                    1.03676532638344962E-03,
                    8.71437918068619115E-04,
                    7.16080155297701002E-04,
                    5.72637002558129372E-04,
                    4.42089819465802277E-04,
                    3.24724948503090564E-04,
                    2.20342042730246599E-04,
                    1.28412898401353882E-04,
                      4.82005924552095464E-05 )  )
                FLOAT (18);
   DECLARE ( alfa(180) )  FLOAT (18);
   DECLARE ( beta1(30)  STATIC INITIAL (
                    1.79988721413553309E-02,
                    5.59964911064388073E-03,
                    2.88501402231132779E-03,
                    1.80096606761053941E-03,
                    1.24753110589199202E-03,
                    9.22878876572938311E-04,
                    7.14430421727287357E-04,
                    5.71787281789704872E-04,
                    4.69431007606481533E-04,
                    3.93232835462916638E-04,
                    3.34818889318297664E-04,
                    2.88952148495751517E-04,
                    2.52211615549573284E-04,
                    2.22280580798883327E-04,
                    1.97541838033062524E-04,
                    1.76836855019718004E-04,
                    1.59316899661821081E-04,
                    1.44347930197333986E-04,
                    1.31448068119965379E-04,
                    1.20245444949302884E-04,
                    1.10449144504599392E-04,
                    1.01828770740567258E-04,
                    9.41998224204237509E-05,
                    8.74130545753834437E-05,
                    8.13466262162801467E-05,
                    7.59002269646219339E-05,
                    7.09906300634153481E-05,
                    6.65482874842468183E-05,
                    6.25146958969275078E-05,
                    5.88403394426251749E-05 )  )
                FLOAT (18);
   DECLARE ( beta2(30)  STATIC INITIAL (
                   -1.49282953213429172E-03,
                   -8.78204709546389328E-04,
                   -5.02916549572034614E-04,
                   -2.94822138512746025E-04,
                   -1.75463996970782828E-04,
                   -1.04008550460816434E-04,
                   -5.96141953046457895E-05,
                   -3.12038929076098340E-05,
                   -1.26089735980230047E-05,
                   -2.42892608575730389E-07,
                    8.05996165414273571E-06,
                    1.36507009262147391E-05,
                    1.73964125472926261E-05,
                    1.98672978842133780E-05,
                    2.14463263790822639E-05,
                    2.23954659232456514E-05,
                    2.28967783814712629E-05,
                    2.30785389811177817E-05,
                    2.30321976080909144E-05,
                    2.28236073720348722E-05,
                    2.25005881105292418E-05,
                    2.20981015361991429E-05,
                    2.16418427448103905E-05,
                    2.11507649256220843E-05,
                    2.06388749782170737E-05,
                    2.01165241997081666E-05,
                    1.95913450141179244E-05,
                    1.90689367910436740E-05,
                    1.85533719641636667E-05,
                    1.80475722259674218E-05 )  )
                FLOAT (18);
   DECLARE ( beta3(30)  STATIC INITIAL (
                    5.52213076721292790E-04,
                    4.47932581552384646E-04,
                    2.79520653992020589E-04,
                    1.52468156198446602E-04,
                    6.93271105657043598E-05,
                    1.76258683069991397E-05,
                   -1.35744996343269136E-05,
                   -3.17972413350427135E-05,
                   -4.18861861696693365E-05,
                   -4.69004889379141029E-05,
                   -4.87665447413787352E-05,
                   -4.87010031186735069E-05,
                   -4.74755620890086638E-05,
                   -4.55813058138628452E-05,
                   -4.33309644511266036E-05,
                   -4.09230193157750364E-05,
                   -3.84822638603221274E-05,
                   -3.60857167535410501E-05,
                   -3.37793306123367417E-05,
                   -3.15888560772109621E-05,
                   -2.95269561750807315E-05,
                   -2.75978914828335759E-05,
                   -2.58006174666883713E-05,
                   -2.41308356761280200E-05,
                   -2.25823509518346033E-05,
                   -2.11479656768912971E-05,
                   -1.98200638885294927E-05,
                   -1.85909870801065077E-05,
                   -1.74532699844210224E-05,
                   -1.63997823854497997E-05 )  ) FLOAT (18);

   DECLARE ( beta4(30)  STATIC INITIAL (
                   -4.74617796559959808E-04,
                   -4.77864567147321487E-04,
                   -3.20390228067037603E-04,
                   -1.61105016119962282E-04,
                   -4.25778101285435204E-05,
                    3.44571294294967503E-05,
                    7.97092684075674924E-05,
                    1.03138236708272200E-04,
                    1.12466775262204158E-04,
                    1.13103642108481389E-04,
                    1.08651634848774268E-04,
                    1.01437951597661973E-04,
                    9.29298396593363896E-05,
                    8.40293133016089978E-05,
                    7.52727991349134062E-05,
                    6.69632521975730872E-05,
                    5.92564547323194704E-05,
                    5.22169308826975567E-05,
                    4.58539485165360646E-05,
                    4.01445513891486808E-05,
                    3.50481730031328081E-05,
                    3.05157995034346659E-05,
                    2.64956119950516039E-05,
                    2.29363633690998152E-05,
                    1.97893056664021636E-05,
                    1.70091984636412623E-05,
                    1.45547428261524004E-05,
                    1.23886640995878413E-05,
                    1.04775876076583236E-05,
                    8.79179954978479373E-06 )  ) FLOAT (18);

   DECLARE ( beta5(30)  STATIC INITIAL (
                    7.36465810572578444E-04,
                    8.72790805146193976E-04,
                    6.22614862573135066E-04,
                    2.85998154194304147E-04,
                    3.84737672879366102E-06,
                   -1.87906003636971558E-04,
                   -2.97603646594554535E-04,
                   -3.45998126832656348E-04,
                   -3.53382470916037712E-04,
                   -3.35715635775048757E-04,
                   -3.04321124789039809E-04,
                   -2.66722723047612821E-04,
                   -2.27654214122819527E-04,
                   -1.89922611854562356E-04,
                   -1.55058918599093870E-04,
                   -1.23778240761873630E-04,
                   -9.62926147717644187E-05,
                   -7.25178327714425337E-05,
                   -5.22070028895633801E-05,
                   -3.50347750511900522E-05,
                   -2.06489761035551757E-05,
                   -8.70106096849767054E-06,
                    1.13698686675100290E-06,
                    9.16426474122778849E-06,
                    1.56477785428872620E-05,
                    2.08223629482466847E-05,
                    2.48923381004595156E-05,
                    2.80340509574146325E-05,
                    3.03987774629861915E-05,
                    3.21156731406700616E-05 )  ) FLOAT (18);

   DECLARE ( beta6(30)  STATIC INITIAL (
                   -1.80182191963885708E-03,
                   -2.43402962938042533E-03,
                   -1.83422663549856802E-03,
                   -7.62204596354009765E-04,
                    2.39079475256927218E-04,
                    9.49266117176881141E-04,
                    1.34467449701540359E-03,
                    1.48457495259449178E-03,
                    1.44732339830617591E-03,
                    1.30268261285657186E-03,
                    1.10351597375642682E-03,
                    8.86047440419791759E-04,
                    6.73073208165665473E-04,
                    4.77603872856582378E-04,
                    3.05991926358789362E-04,
                    1.60315694594721630E-04,
                    4.00749555270613286E-05,
                   -5.66607461635251611E-05,
                   -1.32506186772982638E-04,
                   -1.90296187989614057E-04,
                   -2.32811450376937408E-04,
                   -2.62628811464668841E-04,
                   -2.82050469867598672E-04,
                   -2.93081563192861167E-04,
                   -2.97435962176316616E-04,
                   -2.96557334239348078E-04,
                   -2.91647363312090861E-04,
                   -2.83696203837734166E-04,
                   -2.73512317095673346E-04,
                   -2.61750155806768580E-04 )  ) FLOAT (18);

   DECLARE ( beta7(30)  STATIC INITIAL (
                    6.38585891212050914E-03,
                    9.62374215806377941E-03,
                    7.61878061207001043E-03,
                    2.83219055545628054E-03,
                   -2.09841352012720090E-03,
                   -5.73826764216626498E-03,
                   -7.70804244495414620E-03,
                   -8.21011692264844401E-03,
                   -7.65824520346905413E-03,
                   -6.47209729391045177E-03,
                   -4.99132412004966473E-03,
                   -3.45612289713133280E-03,
                   -2.01785580014170775E-03,
                   -7.59430686781961401E-04,
                    2.84173631523859138E-04,
                    1.10891667586337403E-03,
                    1.72901493872728771E-03,
                    2.16812590802684701E-03,
                    2.45357710494539735E-03,
                    2.61281821058334862E-03,
                    2.67141039656276912E-03,
                    2.65203073395980430E-03,
                    2.57411652877287315E-03,
                    2.45389126236094427E-03,
                    2.30460058071795494E-03,
                    2.13684837686712662E-03,
                    1.95896528478870911E-03,
                    1.77737008679454412E-03,
                    1.59690280765839059E-03,
                    1.42111975664438546E-03 )  ) FLOAT (18);

   DECLARE ( beta(210) )  FLOAT (18);
   DECLARE ( gama(30)  STATIC INITIAL (
                    6.29960524947436582E-01,
                    2.51984209978974633E-01,
                    1.54790300415655846E-01,
                    1.10713062416159013E-01,
                    8.57309395527394825E-02,
                    6.97161316958684292E-02,
                    5.86085671893713576E-02,
                    5.04698873536310685E-02,
                    4.42600580689154809E-02,
                    3.93720661543509966E-02,
                    3.54283195924455368E-02,
                    3.21818857502098231E-02,
                    2.94646240791157679E-02,
                    2.71581677112934479E-02,
                    2.51768272973861779E-02,
                    2.34570755306078891E-02,
                    2.19508390134907203E-02,
                    2.06210828235646240E-02,
                    1.94388240897880846E-02,
                    1.83810633800683158E-02,
                    1.74293213231963172E-02,
                    1.65685837786612353E-02,
                    1.57865285987918445E-02,
                    1.50729501494095594E-02,
                    1.44193250839954639E-02,
                    1.38184805735341786E-02,
                    1.32643378994276568E-02,
                    1.27517121970498651E-02,
                    1.22761545318762767E-02,
                    1.18338262398482403E-02 )  ) FLOAT (18);

   DECLARE ( ex1  STATIC INITIAL ( 3.33333333333333333E-01)
           , ex2  STATIC INITIAL ( 6.66666666666666667E-01)
           , hpi  STATIC INITIAL ( 1.57079632679489662E0)
           , pi   STATIC INITIAL ( 3.14159265358979324E0)
           , thpi STATIC INITIAL ( 4.71238898038468986E0) )  FLOAT (18);
   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i) ) COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( ias, ibs, is );
      PUT SKIP DATA ( j, jr, ju );
      PUT SKIP DATA ( k, kmax, kp1 );
      PUT SKIP DATA ( ks, l, lr );
      PUT SKIP DATA ( lrp1, l1, l2, m );
      RESIGNAL;
   END;

/* Associate arrays alfa & beta                                          */

   DO i = 1 to 30;
      alfa(    i) = alfa1(i);
      alfa(i+ 30) = alfa2(i);
      alfa(i+ 60) = alfa3(i);
      alfa(i+ 90) = alfa4(i);
      alfa(i+120) = alfa5(i);
      alfa(i+150) = alfa6(i);
      beta(    i) = beta1(i);
      beta(i+ 30) = beta2(i);
      beta(i+ 60) = beta3(i);
      beta(i+ 90) = beta4(i);
      beta(i+120) = beta5(i);
      beta(i+150) = beta6(i);
      beta(i+180) = beta7(i);
   END;

   rfnu = 1.00000000000000000E0 / fnu;
/*     ZB = Z*COMPLEX(RFNU,0.0)                                            */
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST (Z/FNU TOO SMALL)                                   */
/* ----------------------------------------------------------------------- */
   tstr = REAL(z);
   tsti = IMAG(z);
   test = TINY(0.00000000000000000E0) * 1.00000000000000000E+3;
   ac = fnu * test;
   IF  ABS(tstr) <= ac   &   ABS(tsti) <= ac  THEN
      DO;
         ac = 2 * ABS(LOG(test)) + fnu;
         zeta1 = ac;
         zeta2 = fnu;
         phi = cone;
         arg = cone;
         RETURN;
      END;
   zb = z * rfnu;
   rfnu2 = rfnu * rfnu;
/* ----------------------------------------------------------------------- */
/*     COMPUTE IN THE FOURTH QUADRANT                                    */
/* ----------------------------------------------------------------------- */
   fn13 = fnu ** ex1;
   fn23 = fn13 * fn13;
   rfn13 = 1.00000000000000000E0/fn13;
   w2 = cone - zb * zb;
   aw2 = ABS(w2);
   IF  aw2 > 0.25000000000000000E0 THEN  GO TO L110;
/* ----------------------------------------------------------------------- */
/*     POWER SERIES FOR ABS(W2) <= 0.25                                  */
/* ----------------------------------------------------------------------- */
   k = 1;

   p(1) = cone;
   suma = gama(1);
   ap(1) = 1.00000000000000000E0;
   IF  aw2 >= tol  THEN
      DO;
         DO  k = 2 TO  30;
            p(k) = p(k-1) * w2;
            suma = suma + p(k) * gama(k);
            ap(k) = ap(k-1) * aw2;
                IF  ap(k) < tol THEN  GO TO L20;
         END;
         k = 30;
      END;

L20:
   kmax = k;
   zeta = w2 * suma;
   arg = zeta * fn23;
   za = SQRT(suma);
   zeta2 = SQRT(w2) * fnu;
   zeta1 = zeta2 * (cone + zeta*za*ex2);
   za = za + za;
   phi = SQRT(za) * rfn13;
   IF  ipmtr ^= 1  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     SUM SERIES FOR ASUM AND BSUM                                      */
/* ----------------------------------------------------------------------- */
         sumb = czero;
         DO k = 1 TO  kmax;
            sumb = sumb + p(k)*beta(k);
         END;
         asum = czero;
         bsum = sumb;
         l1 = 0;
         l2 = 30;
         btol = tol * (ABS(REAL(bsum)) + ABS( IMAG(bsum)));
         atol = tol;
         pp = 1.00000000000000000E0;
         ias = 0;
         ibs = 0;
         IF  rfnu2 >= tol  THEN
            DO;
               DO  is = 2 TO  7;
                  atol = atol / rfnu2;
                  pp = pp * rfnu2;
                  IF  ias ^= 1  THEN
                     DO;
                        suma = czero;
                        DO  k = 1 TO  kmax;
                           m = l1 + k;
                           suma = suma + p(k) * alfa(m);
                           IF  ap(k) < atol THEN
                              LEAVE;
                        END;
                        asum = asum + suma * pp;
                        IF  pp < tol THEN
                           ias = 1;
                     END;
                  IF  ibs ^= 1  THEN
                     DO;
                        sumb = czero;
                        DO  k = 1 TO  kmax;
                           m = l2 + k;
                           sumb = sumb + p(k) * beta(m);
                           IF  ap(k) < atol THEN
                              LEAVE;
                        END;
                        bsum = bsum + sumb * pp;
                        IF  pp < btol THEN
                           ibs = 1;
                     END;
                  IF  ias  = 1   &   ibs  = 1 THEN
                     LEAVE;
                  l1 = l1 + 30;
                  l2 = l2 + 30;
               END;
            END;

         asum = asum + cone;
         pp = rfnu * rfn13;
         bsum = bsum * pp;
      END;

L100:
   RETURN;
/* ----------------------------------------------------------------------- */
/*     ABS(W2) > 0.25                                                    */
/* ----------------------------------------------------------------------- */
L110:
   w  = SQRT(w2);
   wr = REAL(w);
   wi = IMAG(w);
   IF  wr < 0 THEN
      wr = 0;
   IF  wi < 0 THEN
      wi = 0;
   w = COMPLEX(wr, wi);
   za = (cone+w) / zb;
   zc = LOG(za);
   zcr = REAL(zc);
   zci = IMAG(zc);
   IF  zci < 0 THEN
      zci = 0;
   IF  zci > hpi THEN
      zci = hpi;
   IF  zcr < 0 THEN
      zcr = 0;
   zc = COMPLEX(zcr, zci);
   zth = (zc-w) * 1.50000000000000000E0;
   cfnu = COMPLEX(fnu, 0);
   zeta1 = zc * cfnu;
   zeta2 = w * cfnu;
   azth = ABS(zth);
   zthr = REAL(zth);
   zthi = IMAG(zth);
   ang = thpi;
   IF  zthr < 0  |   zthi >= 0  THEN
      DO;
         ang = hpi;
         IF  zthr ^= 0  THEN
            DO;
               ang = ATAN(zthi/zthr);
               IF  zthr < 0 THEN
                  ang = ang + pi;
            END;
      END;
   pp = azth ** ex2;
   ang = ang * ex2;
   zetar = pp * COS(ang);
   zetai = pp * SIN(ang);
   IF  zetai < 0 THEN
      zetai = 0;
   zeta = COMPLEX(zetar, zetai);
   arg = zeta * fn23;
   rtzta = zth / zeta;
   za = rtzta / w;
   phi = SQRT(za+za) * rfn13;
   IF  ipmtr  = 1 THEN  GO TO L100;
   tfn  = COMPLEX(rfnu, 0) / w;
   rzth = COMPLEX(rfnu, 0) / zth;
   zc = rzth * ar(2);
   t2 = cone / w2;
   up(2) = (t2*c(2) + c(3)) * tfn;
   bsum = up(2) + zc;
   asum = czero;
   IF  rfnu >= tol  THEN
      DO;
         przth = rzth;
         ptfn = tfn;
         up(1) = cone;
         pp = 1;
         bsumr = REAL(bsum);
         bsumi = IMAG(bsum);
         btol = tol * (ABS(bsumr) + ABS(bsumi));
         ks = 0;
         kp1 = 2;
         l = 3;
         ias = 0;
         ibs = 0;
         DO  lr = 2 TO  12 BY  2;
            lrp1 = lr + 1;
/* ----------------------------------------------------------------------- */
/*     COMPUTE TWO ADDITIONAL CR, DR, AND UP FOR TWO MORE TERMS IN       */
/*     NEXT SUMA AND SUMB                                                */
/* ----------------------------------------------------------------------- */
            DO  k = lr TO  lrp1;
               ks = ks + 1;
               kp1 = kp1 + 1;
               l = l + 1;
               za = COMPLEX(c(l), 0);
               DO  j = 2 TO  kp1;
                  l = l + 1;
                  za = za * t2 + c(l);
               END;
               ptfn = ptfn * tfn;
               up(kp1) = ptfn * za;
               cr(ks) = przth * br(ks+1);
               przth = przth * rzth;
               dr(ks) = przth * ar(ks+2);
            END;
            pp = pp * rfnu2;
            IF  ias ^= 1  THEN
               DO;
                  suma = up(lrp1);
                  ju = lrp1;
                  DO  jr = 1 TO  lr;
                     ju = ju - 1;
                     suma = suma + cr(jr) * up(ju);
                  END;
                  asum = asum + suma;
                  asumr = REAL(asum);
                  asumi = IMAG(asum);
                  test = ABS(asumr) + ABS(asumi);
                  IF  pp < tol   &   test < tol THEN
                     ias = 1;
               END;
            IF  ibs ^= 1  THEN
               DO;
                  sumb = up(lr+2) + up(lrp1) * zc;
                  ju = lrp1;
                  DO  jr = 1 TO  lr;
                     ju = ju - 1;
                     sumb = sumb + dr(jr) * up(ju);
                  END;
                  bsum = bsum + sumb;
                  bsumr = REAL(bsum);
                  bsumi = IMAG(bsum);
                  test = ABS(bsumr) + ABS(bsumi);
                  IF  pp < btol   &   test < tol THEN
                     ibs = 1;
               END;
            IF  ias  = 1   &   ibs  = 1 THEN  GO TO L170;
         END;
      END;

L170:
   asum = asum + cone;
   bsum = -bsum * rfn13 / rtzta;
   GO TO L100;
   END cunhj;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cseri: PROCEDURE (z, fnu, kode, n, y, nz, tol, elim, alim) OPTIONS (REORDER);   
/* ***BEGIN PROLOGUE  CSERI                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*     CSERI COMPUTES THE I BESSEL FUNCTION FOR REAL(Z) >= 0.0 BY        */
/*     MEANS OF THE POWER SERIES FOR LARGE ABS(Z) IN THE                 */
/*     REGION ABS(Z) <= 2*SQRT(FNU+1). NZ=0 IS A NORMAL RETURN.          */
/*     NZ > 0 MEANS THAT THE LAST NZ COMPONENTS WERE SET TO ZERO         */
/*     DUE TO UNDERFLOW. NZ < 0 MEANS UNDERFLOW OCCURRED, BUT THE        */
/*     CONDITION ABS(Z) <= 2*SQRT(FNU+1) WAS VIOLATED AND THE            */
/*     COMPUTATION MUST BE COMPLETED IN ANOTHER ROUTINE WITH N=N-ABS(NZ). */

/* ***ROUTINES CALLED  CUCHK,GAMLN,R1MACH                                */
/* ***END PROLOGUE  CSERI                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( ak1, ck, coef, crsc, cz, hz, rz, s1, s2, w(2) )  COMPLEX FLOAT (18);
   DECLARE ( aa, acz, ak, arm, ascle, atol, az, dfnu, fnup, rak1, rs, rtr1, 
      s, ss, x )  FLOAT (18);
   DECLARE ( i, ib, iflag, il, k, l, m, nn, nw )  FIXED BINARY (31);
   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i) ) COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, ib, iflag );
      PUT SKIP DATA ( il, k, l );
      PUT SKIP DATA ( m, nn, nw );
      RESIGNAL;
   END;

   nz = 0;
   az = ABS(z);
   IF  az ^= 0  THEN
      DO;
         x = REAL(z);
         arm = 1.00000000000000000E+3 * TINY(0.00000000000000000E0);
         rtr1 = SQRT(arm);
         crsc = 1+0i;
         iflag = 0;
         IF  az >= arm  THEN
            DO;
               hz = z * 0.50000000000000000E0;
               cz = czero;
               IF  az > rtr1 THEN
                  cz = hz * hz;
               acz = ABS(cz);
               nn = n;
               ck = LOG(hz);

L10:
               dfnu = fnu + (nn-1);
               fnup = dfnu + 1.00000000000000000E0;
/* ----------------------------------------------------------------------- */
/*     UNDERFLOW TEST                                                    */
/* ----------------------------------------------------------------------- */
               ak1 = ck * dfnu;
               ak = gamln(fnup);
               ak1 = ak1 - ak;
               IF  kode  = 2 THEN
                  ak1 = ak1 - x;
               rak1 = REAL(ak1);
               IF  rak1 > -elim THEN  GO TO L30;

L20:
               nz = nz + 1;
               y(nn) = czero;
               IF  acz > dfnu THEN  GO TO L120;
               nn = nn - 1;
               IF  nn  = 0 THEN
                  RETURN;
               GO TO L10;

L30:
               IF  rak1 <= -alim  THEN
                  DO;
                     iflag = 1;
                     ss = 1.00000000000000000E0 / tol;
                     crsc = COMPLEX(tol, 0);
                     ascle = arm * ss;
                  END;
               ak = IMAG(ak1);
               aa = EXP(rak1);
               IF  iflag  = 1 THEN
                  aa = aa * ss;
               coef = aa * COMPLEX(COS(ak), SIN(ak) );
               atol = tol * acz / fnup;
               il = MIN(2,nn);
               DO  i = 1 TO  il;
                  dfnu = fnu + (nn-i);
                  fnup = dfnu + 1.00000000000000000E0;
                  s1 = cone;
                  IF  acz >= tol*fnup  THEN
                     DO;
                        ak1 = cone;
                        ak = fnup + 2.00000000000000000E0;
                        s = fnup;
                        aa = 2.00000000000000000E0;

L40:
                        rs = 1.00000000000000000E0 / s;
                        ak1 = ak1 * cz * rs;
                        s1 = s1 + ak1;
                        s = s + ak;
                        ak = ak + 2.00000000000000000E0;
                        aa = aa * acz * rs;
                        IF  aa > atol THEN  GO TO L40;
                     END;
                  m = nn - i + 1;
                  s2 = s1 * coef;
                  w(i) = s2;
                  IF  iflag ^= 0  THEN
                     DO;
                        CALL cuchk(s2, nw, ascle, tol);
                        IF  nw ^= 0 THEN  GO TO L20;
                     END;
                  y(m) = s2 * crsc;
                  IF  i ^= il THEN
                     coef = coef * dfnu / hz;
               END;
               IF  nn <= 2 THEN
                  RETURN;
               k = nn - 2;
               ak = k;
               rz = (cone+cone) / z;
               IF  iflag  = 1 THEN  GO TO L80;
               ib = 3;

L60:
               DO  i = ib TO  nn;
                  y(k) = (ak+fnu) * rz * y(k+1) + y(k+2);
                  ak = ak - 1.00000000000000000E0;
                  k = k - 1;
               END;
               RETURN;
/* ----------------------------------------------------------------------- */
/*     RECUR BACKWARD WITH SCALED VALUES                                 */
/* ----------------------------------------------------------------------- */
/*     EXP(-ALIM)=EXP(-ELIM)/TOL=APPROX. ONE PRECISION ABOVE THE         */
/*     UNDERFLOW LIMIT = ASCLE = TINY(0.0)*CSCL*1.0E+3                   */
/* ----------------------------------------------------------------------- */
L80:
               s1 = w(1);
               s2 = w(2);
               DO  l = 3 TO  nn;
                  ck = s2;
                  s2 = s1 + (ak+fnu) * rz * s2;
                  s1 = ck;
                  ck = s2 * crsc;
                  y(k) = ck;
                  ak = ak - 1.00000000000000000E0;
                  k = k - 1;
                  IF  ABS(ck) > ascle THEN  GO TO L100;
               END;
               RETURN;

L100:
               ib = l + 1;
               IF  ib > nn THEN
                  RETURN;
               GO TO L60;
            END;
         nz = n;
         IF  fnu  = 0 THEN
            nz = nz - 1;
      END;
   y(1) = czero;
   IF  fnu  = 0 THEN
      y(1) = cone;
   IF  n  = 1 THEN
      RETURN;
   DO i = 2 TO n; y(i) = czero; END;
   RETURN;
/* ----------------------------------------------------------------------- */
/*     RETURN WITH NZ < 0 IF ABS(Z*Z/4) > FNU+N-NZ-1 COMPLETE            */
/*     THE CALCULATION IN CBINU WITH N=N-ABS(NZ)                         */
/* ----------------------------------------------------------------------- */
L120:
   nz = -nz;
   RETURN;
   END cseri;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
casyi: PROCEDURE (z, fnu, kode, n, y, nz, rl, tol, elim, alim)
          OPTIONS (REORDER);                                                                               
/* ***BEGIN PROLOGUE  CASYI                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*     CASYI COMPUTES THE I BESSEL FUNCTION FOR REAL(Z) >= 0.0 BY        */
/*     MEANS OF THE ASYMPTOTIC EXPANSION FOR LARGE ABS(Z) IN THE         */
/*     REGION ABS(Z) > MAX(RL,FNU*FNU/2).  NZ=0 IS A NORMAL RETURN.      */
/*     NZ < 0 INDICATES AN OVERFLOW ON KODE=1.                           */

/* ***ROUTINES CALLED  R1MACH                                            */
/* ***END PROLOGUE  CASYI                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( rl )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( ak1, ck, cs1, cs2, cz, dk, ez, p1, rz, s2 )  COMPLEX FLOAT (18);
   DECLARE ( aa, acz, aez, ak, arg, arm, atol, az, bb, bk, dfnu, dnu2, fdn, 
      rtr1, s, sgn, sqk, x, yy )  FLOAT (18);
   DECLARE ( i, ib, il, inu, j, jl, k, koded, m, nn )  FIXED BINARY (31);

   DECLARE ( pi   STATIC INITIAL ( 3.14159265358979324E0)
           , rtpi STATIC INITIAL ( 0.159154943091895336E0)
               )  FLOAT (18);
/* rtpi = reciprocal of 2.pi                                             */
   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i) ) COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, ib, il );
      PUT SKIP DATA ( inu, j, jl );
      PUT SKIP DATA ( k, koded, m, nn );
      RESIGNAL;
   END;

   nz = 0;
   az = ABS(z);
   x = REAL(z);
   arm = 1.00000000000000000E+3 * TINY(0.00000000000000000E0);
   rtr1 = SQRT(arm);
   il = MIN(2,n);
   dfnu = fnu + (n-il);
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST                                                     */
/* ----------------------------------------------------------------------- */
   ak1 = rtpi / z;
   ak1 = SQRT(ak1);
   cz = z;
   IF  kode  = 2 THEN
      cz = z - x;
   acz = REAL(cz);
   IF  ABS(acz) <= elim  THEN
      DO;
         dnu2 = dfnu + dfnu;
         koded = 1;
         IF  ^ (ABS(acz) > alim   &   n > 2)  THEN
            DO;
               koded = 0;
               ak1 = ak1 * EXP(cz);
            END;
         fdn = 0.00000000000000000E0;
         IF  dnu2 > rtr1 THEN
            fdn = dnu2 * dnu2;
         ez = z * 8.00000000000000000E0;
/* ----------------------------------------------------------------------- */
/*     WHEN Z IS IMAGINARY, THE ERROR TEST MUST BE MADE RELATIVE TO THE  */
/*     FIRST RECIPROCAL POWER SINCE THIS IS THE LEADING TERM OF THE      */
/*     EXPANSION FOR THE IMAGINARY PART.                                 */
/* ----------------------------------------------------------------------- */
         aez = 8.00000000000000000E0 * az;
         s = tol / aez;
         jl = rl + rl + 2;
         yy =  IMAG(z);
         p1 = czero;
         IF  yy ^= 0.00000000000000000E0  THEN
            DO;
/* ----------------------------------------------------------------------- */
/*     CALCULATE EXP(PI*(0.5+FNU+N-IL)*I) TO MINIMIZE LOSSES OF          */
/*     SIGNIFICANCE WHEN FNU OR N IS LARGE                               */
/* ----------------------------------------------------------------------- */
               inu = fnu;
               arg = (fnu - inu) * pi;
               inu = inu + n - il;
               ak = -SIN(arg);
               bk = COS(arg);
               IF  yy < 0 THEN
                  bk = -bk;
               p1 = COMPLEX(ak, bk);
               IF  REM(inu,2)  = 1 THEN
                  p1 = -p1;
            END;
         DO  k = 1 TO  il;
            sqk = fdn - 1.00000000000000000E0;
            atol = s * ABS(sqk);
            sgn = 1.00000000000000000E0;
            cs1 = cone;
            cs2 = cone;
            ck = cone;
            ak = 0.00000000000000000E0;
            aa = 1.00000000000000000E0;
            bb = aez;
            dk = ez;
            DO  j = 1 TO  jl;
               ck = ck * sqk / dk;
               cs2 = cs2 + ck;
               sgn = -sgn;
               cs1 = cs1 + ck * sgn;
               dk = dk + ez;
               aa = aa * ABS(sqk) / bb;
               bb = bb + aez;
               ak = ak + 8.00000000000000000E0;
               sqk = sqk - ak;
               IF  aa <= atol THEN  GO TO L20;
            END;
            GO TO L60;

L20:
            s2 = cs1;
            IF  x+x < elim THEN
               s2 = s2 + p1 * cs2 * EXP(-z-z);
            fdn = fdn + 8 * dfnu + 4;
            p1 = -p1;
            m = n - il + k;
            y(m) = s2 * ak1;
         END;
         IF  n <= 2 THEN
            RETURN;
         nn = n;
         k = nn - 2;
         ak = k;
         rz = (cone+cone) / z;
         ib = 3;
         DO  i = ib TO  nn;
            y(k) = COMPLEX(ak+fnu, 0) * rz * y(k+1) + y(k+2);
            ak = ak - 1;
            k = k - 1;
         END;
         IF  koded  = 0 THEN
            RETURN;
         ck = EXP(cz);
         DO i = 1 TO nn; y(i) = y(i) * ck; END;
         RETURN;
      END;
   nz = -1;
   RETURN;

L60:
   nz = -2;
   RETURN;
   END casyi;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbunk: PROCEDURE (z, fnu, kode, mr, n, y, nz, tol, elim, alim)
          OPTIONS (REORDER);                                                                               
/* ***BEGIN PROLOGUE  CBUNK                                              */
/* ***REFER TO  CBESK,CBESH                                              */

/*     CBUNK COMPUTES THE K BESSEL FUNCTION FOR FNU > FNUL.              */
/*     ACCORDING TO THE UNIFORM ASYMPTOTIC EXPANSION FOR K(FNU,Z)        */
/*     IN CUNK1 AND THE EXPANSION FOR H(2,FNU,Z) IN CUNK2                */

/* ***ROUTINES CALLED  CUNK1,CUNK2                                       */
/* ***END PROLOGUE  CBUNK                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( mr )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( ax, ay, xx, yy )  FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( mr, n, nz );
      RESIGNAL;
   END;

   nz = 0;
   xx = REAL(z);
   yy = IMAG(z);
   ax = ABS(xx) * 1.73210000000000000E0;
   ay = ABS(yy);
   IF  ay <= ax  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     ASYMPTOTIC EXPANSION FOR K(FNU,Z) FOR LARGE FNU APPLIED IN        */
/*     -PI/3 <= ARG(Z) <= PI/3                                           */
/* ----------------------------------------------------------------------- */
         CALL cunk1(z, fnu, kode, mr, n, y, nz, tol, elim, alim);
      END;
   ELSE
      DO;
/* ----------------------------------------------------------------------- */
/*     ASYMPTOTIC EXPANSION FOR H(2,FNU,Z*EXP(M*HPI)) FOR LARGE FNU      */
/*     APPLIED IN PI/3 < ABS(ARG(Z)) <= PI/2 WHERE M=+I OR -I AND HPI=PI/2 */
/* ----------------------------------------------------------------------- */
         CALL cunk2(z, fnu, kode, mr, n, y, nz, tol, elim, alim);
      END;
   RETURN;
   END cbunk;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cunk1: PROCEDURE (z, fnu, kode, mr, n, y, nz, tol, elim, alim)
          OPTIONS (REORDER);                                                                               
/* ***BEGIN PROLOGUE  CUNK1                                              */
/* ***REFER TO  CBESK                                                    */

/*     CUNK1 COMPUTES K(FNU,Z) AND ITS ANALYTIC CONTINUATION FROM THE    */
/*     RIGHT HALF PLANE TO THE LEFT HALF PLANE BY MEANS OF THE           */
/*     UNIFORM ASYMPTOTIC EXPANSION.                                     */
/*     MR INDICATES THE DIRECTION OF ROTATION FOR ANALYTIC CONTINUATION. */
/*     NZ=-1 MEANS AN OVERFLOW WILL OCCUR                                */

/* ***ROUTINES CALLED  CS1S2,CUCHK,CUNIK,R1MACH                          */
/* ***END PROLOGUE  CUNK1                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( mr )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( cfn, ck, crsc, cs, cscl, csgn, cspn, csr(3), css(3), cwrk(16,3),
      cy(2), c1, c2, phi(2), rz, sum(2), s1, s2, zeta1(2), zeta2(2), zr, phid,
      zeta1d, zeta2d, sumd )  COMPLEX FLOAT (18);
   DECLARE ( ang, aphi, asc, ascle, bry(3), cpn, c2i, c2m, c2r, fmr, fn, fnf, 
      rs1, sgn, spn, x )  FLOAT (18);
   DECLARE ( i, ib, iflag, ifn, il, init(2), inu, iuf, k, kdflg, kflag, kk, 
      m, nw, j, ipard, initd, ic )  FIXED BINARY (31);
   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i) ) COMPLEX FLOAT (18);
   DECLARE ( pi  STATIC INITIAL ( 3.14159265358979324E0) )  FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, ib, iflag, ifn, il, init, inu, iuf, k, kdflg, kflag, kk );
      PUT SKIP DATA ( m, nw, j, ipard, initd, ic );
      RESIGNAL;
   END;

   kdflg = 1;
   nz = 0;
/* ----------------------------------------------------------------------- */
/*     EXP(-ALIM)=EXP(-ELIM)/TOL=APPROX. ONE PRECISION GREATER THAN      */
/*     THE UNDERFLOW LIMIT                                               */
/* ----------------------------------------------------------------------- */
   cscl = 1.00000000000000000E0/tol;
   crsc = tol;
   css(1) = cscl;
   css(2) = cone;
   css(3) = crsc;
   csr(1) = crsc;
   csr(2) = cone;
   csr(3) = cscl;
   bry(1) = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
   bry(2) = 1.00000000000000000E0 / bry(1);
   bry(3) = HUGE(0.00000000000000000E0);
   x = REAL(z);
   zr = z;
   IF  x < 0 THEN
      zr = -z;
   j = 2;

L05:
   DO  i = 1 TO  n;
/* ----------------------------------------------------------------------- */
/*     J FLIP FLOPS BETWEEN 1 AND 2 IN J = 3 - J                         */
/* ----------------------------------------------------------------------- */
      j = 3 - j;
      fn = fnu + (i-1);
      init(j) = 0;
      CALL cunik(zr, fn, 2, 0, tol, init(j), phi(j), zeta1(j), zeta2(j), sum(j),
         cwrk(*,j));
      IF  kode ^= 1  THEN
         DO;
            cfn = fn;
            s1 = zeta1(j) - cfn * (cfn/(zr + zeta2(j)));
         END;
      ELSE
         DO;
            s1 = zeta1(j) - zeta2(j);
         END;
/* ----------------------------------------------------------------------- */
/*     TEST FOR UNDERFLOW AND OVERFLOW                                   */
/* ----------------------------------------------------------------------- */
      rs1 = REAL(s1);
      IF  ABS(rs1) <= elim  THEN
         DO;
            IF  kdflg  = 1 THEN
               kflag = 2;
            IF  ABS(rs1) >= alim  THEN
               DO;
/* ----------------------------------------------------------------------- */
/*     REFINE  TEST AND SCALE                                            */
/* ----------------------------------------------------------------------- */
                  aphi = ABS(phi(j));
                  rs1 = rs1 + LOG(aphi);
                  IF  ABS(rs1) > elim THEN  GO TO L10;
                  IF  kdflg  = 1 THEN
                     kflag = 1;
                  IF  rs1 >= 0  THEN
                     DO;
                        IF  kdflg  = 1 THEN
                           kflag = 3;
                     END;
               END;
/* ----------------------------------------------------------------------- */
/*     SCALE S1 TO KEEP INTERMEDIATE ARITHMETIC ON SCALE NEAR            */
/*     EXPONENT EXTREMES                                                 */
/* ----------------------------------------------------------------------- */
            s2 = phi(j) * sum(j);
            c2r = REAL(s1);
            c2i = IMAG(s1);
            c2m = EXP(c2r) * REAL(css(kflag) );
            s1 = c2m * COMPLEX(COS(c2i), SIN(c2i) );
            s2 = s2 * s1;
            IF  kflag  = 1  THEN
               DO;
                  CALL cuchk(s2, nw, bry(1), tol);
                  IF  nw ^= 0 THEN  GO TO L10;
               END;
            cy(kdflg) = s2;
            y(i) = s2 * csr(kflag);
            IF  kdflg  = 2 THEN  GO TO L30;
            kdflg = 2;
            ITERATE L05;
         END;

L10:
      IF  rs1 > 0 THEN  GO TO L150;
/* ----------------------------------------------------------------------- */
/*     FOR X < 0.0, THE I FUNCTION TO BE ADDED WILL OVERFLOW             */
/* ----------------------------------------------------------------------- */
      IF  x < 0 THEN  GO TO L150;
      kdflg = 1;
      y(i) = czero;
      nz = nz + 1;
      IF  i ^= 1  THEN
         DO;
            IF  y(i-1) ^= czero  THEN
               DO;
                  y(i-1) = czero;
                  nz = nz + 1;
               END;
         END;
   END;
   i = n;

L30:
   rz = 2 / zr;
   ck = fn * rz;
   ib = i + 1;
   IF  n >= ib  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     TEST LAST MEMBER FOR UNDERFLOW AND OVERFLOW, SET SEQUENCE TO ZERO */
/*     ON UNDERFLOW                                                      */
/* ----------------------------------------------------------------------- */
         fn = fnu + (n-1);
         ipard = 1;
         IF  mr ^= 0 THEN
            ipard = 0;
         initd = 0;
         CALL cunik(zr, fn, 2, ipard, tol, initd, phid, zeta1d, zeta2d, sumd,
            cwrk(*,3));
         IF  kode ^= 1  THEN
            DO;
               cfn = fn;
               s1 = zeta1d - cfn * (cfn/(zr + zeta2d));
            END;
         ELSE
            DO;
               s1 = zeta1d - zeta2d;
            END;
         rs1 = REAL(s1);
         IF  ABS(rs1) <= elim  THEN
            DO;
               IF  ABS(rs1) < alim THEN  GO TO L50;
/* ----------------------------------------------------------------------- */
/*     REFINE ESTIMATE AND TEST                                          */
/* ----------------------------------------------------------------------- */
               aphi = ABS(phid);
               rs1 = rs1 + LOG(aphi);
               IF  ABS(rs1) < elim THEN  GO TO L50;
            END;
         IF  rs1 > 0 THEN  GO TO L150;
/* ----------------------------------------------------------------------- */
/*     FOR X < 0.0, THE I FUNCTION TO BE ADDED WILL OVERFLOW             */
/* ----------------------------------------------------------------------- */
         IF  x < 0 THEN  GO TO L150;
         nz = n;
         y(*) = czero;
         RETURN;
/* ----------------------------------------------------------------------- */
/*     RECUR FORWARD FOR REMAINDER OF THE SEQUENCE                       */
/* ----------------------------------------------------------------------- */
L50:
         s1 = cy(1);
         s2 = cy(2);
         c1 = csr(kflag);
         ascle = bry(kflag);
         DO  i = ib TO  n;
            c2 = s2;
            s2 = ck * s2 + s1;
            s1 = c2;
            ck = ck + rz;
            c2 = s2 * c1;
            y(i) = c2;
            IF  kflag < 3  THEN
               DO;
                  c2r = REAL(c2);
                  c2i = IMAG(c2);
                  c2r = ABS(c2r);
                  c2i = ABS(c2i);
                  c2m = MAX(c2r,c2i);
                  IF  c2m > ascle  THEN
                     DO;
                        kflag = kflag + 1;
                        ascle = bry(kflag);
                        s1 = s1 * c1;
                        s2 = c2;
                        s1 = s1 * css(kflag);
                        s2 = s2 * css(kflag);
                        c1 = csr(kflag);
                     END;
               END;
         END;
      END;
   IF  mr  = 0 THEN
      RETURN;
/* ----------------------------------------------------------------------- */
/*     ANALYTIC CONTINUATION FOR RE(Z) < 0.0                             */
/* ----------------------------------------------------------------------- */
   nz = 0;
   fmr = mr;
   sgn = -SIGN(pi, fmr);
/* ----------------------------------------------------------------------- */
/*     CSPN AND CSGN ARE COEFF OF K AND I FUNCIONS RESP.                 */
/* ----------------------------------------------------------------------- */
   csgn = COMPLEX(0, sgn);
   inu = fnu;
   fnf = fnu - inu;
   ifn = inu + n - 1;
   ang = fnf * sgn;
   cpn = COS(ang);
   spn = SIN(ang);
   cspn = COMPLEX(cpn, spn);
   IF  REM(ifn,2)  = 1 THEN
      cspn = -cspn;
   asc = bry(1);
   kk = n;
   iuf = 0;
   kdflg = 1;
   ib = ib - 1;
   ic = ib - 1;

L65:
   DO  k = 1 TO  n;
      fn = fnu + (kk-1);
/* ----------------------------------------------------------------------- */
/*     LOGIC TO SORT OUT CASES WHOSE PARAMETERS WERE SET FOR THE K       */
/*     FUNCTION ABOVE                                                    */
/* ----------------------------------------------------------------------- */
      m = 3;
        IF  n > 2 THEN  GO TO L80;

L70:
      initd = init(j);
      phid = phi(j);
      zeta1d = zeta1(j);
      zeta2d = zeta2(j);
      sumd = sum(j);
      m = j;
      j = 3 - j;
      GO TO L90;

L80:
      IF  ^ (kk  = n   &   ib < n)  THEN
         DO;
            IF  kk  = ib  |   kk  = ic THEN  GO TO L70;
            initd = 0;
         END;

L90:
      CALL cunik(zr, fn, 1, 0, tol, initd, phid, zeta1d, zeta2d, sumd,
         cwrk(*,m));
      IF  kode ^= 1  THEN
         DO;
            cfn = fn;
            s1 = -zeta1d + cfn * (cfn/(zr + zeta2d));
         END;
      ELSE
         DO;
            s1 = -zeta1d + zeta2d;
         END;
/* ----------------------------------------------------------------------- */
/*     TEST FOR UNDERFLOW AND OVERFLOW                                   */
/* ----------------------------------------------------------------------- */
      rs1 = REAL(s1);
      IF  ABS(rs1) > elim THEN  GO TO L110;
      IF  kdflg  = 1 THEN
         iflag = 2;
      IF  ABS(rs1) >= alim  THEN
         DO;
/* ----------------------------------------------------------------------- */
/*     REFINE  TEST AND SCALE                                            */
/* ----------------------------------------------------------------------- */
            aphi = ABS(phid);
            rs1 = rs1 + LOG(aphi);
            IF  ABS(rs1) > elim THEN  GO TO L110;
            IF  kdflg  = 1 THEN
               iflag = 1;
            IF  rs1 >= 0  THEN
               DO;
                  IF  kdflg  = 1 THEN
                     iflag = 3;
               END;
         END;
      s2 = csgn * phid * sumd;
      c2r = REAL(s1);
      c2i = IMAG(s1);
      c2m = EXP(c2r) * REAL(css(iflag));
      s1 = c2m * COMPLEX(COS(c2i), SIN(c2i) );
      s2 = s2 * s1;
      IF  iflag  = 1  THEN
         DO;
            CALL cuchk(s2, nw, bry(1), tol);
            IF  nw ^= 0 THEN
               s2 = 0;
         END;

L100:
      cy(kdflg) = s2;
      c2 = s2;
      s2 = s2 * csr(iflag);
/* ----------------------------------------------------------------------- */
/*     ADD I AND K FUNCTIONS, K SEQUENCE IN Y(I), I=1,N                  */
/* ----------------------------------------------------------------------- */
      s1 = y(kk);
      IF  kode ^= 1  THEN
         DO;
            CALL cs1s2(zr, s1, s2, nw, asc, alim, iuf);
            nz = nz + nw;
         END;
      y(kk) = s1 * cspn + s2;
      kk = kk - 1;
      cspn = -cspn;
      IF  c2  = czero  THEN
         DO;
            kdflg = 1;
            ITERATE L65;
         END;
      IF  kdflg  = 2 THEN  GO TO L130;
      kdflg = 2;
      ITERATE;

L110:
      IF  rs1 > 0 THEN  GO TO L150;
      s2 = czero;
      GO TO L100;
   END;
   k = n;

L130:
   il = n - k;
   IF  il  = 0 THEN
      RETURN;
/* ----------------------------------------------------------------------- */
/*     RECUR BACKWARD FOR REMAINDER OF I SEQUENCE AND ADD IN THE         */
/*     K FUNCTIONS, SCALING THE I SEQUENCE DURING RECURRENCE TO KEEP     */
/*     INTERMEDIATE ARITHMETIC ON SCALE NEAR EXPONENT EXTREMES.          */
/* ----------------------------------------------------------------------- */
   s1 = cy(1);
   s2 = cy(2);
   cs = csr(iflag);
   ascle = bry(iflag);
   fn = inu + il;
   DO  i = 1 TO  il;
      c2 = s2;
      s2 = s1 + (fn + fnf) * rz * s2;
      s1 = c2;
      fn = fn - 1.00000000000000000E0;
      c2 = s2 * cs;
      ck = c2;
      c1 = y(kk);
      IF  kode ^= 1  THEN
         DO;
            CALL cs1s2(zr, c1, c2, nw, asc, alim, iuf);
            nz = nz + nw;
         END;
      y(kk) = c1 * cspn + c2;
      kk = kk - 1;
      cspn = -cspn;
      IF  iflag < 3  THEN
         DO;
            c2r = REAL(ck);
            c2i = IMAG(ck);
            c2r = ABS(c2r);
            c2i = ABS(c2i);
            c2m = MAX(c2r, c2i);
            IF  c2m > ascle  THEN
               DO;
                  iflag = iflag + 1;
                  ascle = bry(iflag);
                  s1 = s1 * cs;
                  s2 = ck;
                  s1 = s1 * css(iflag);
                  s2 = s2 * css(iflag);
                  cs = csr(iflag);
               END;
         END;
   END;
   RETURN;

L150:
   nz = -1;
   RETURN;

DECLARE SIGN GENERIC (DSIGN WHEN (FLOAT, FLOAT),
   ISIGN WHEN (FIXED BINARY, FIXED BINARY) );
ISIGN: PROCEDURE (X, Y) RETURNS (FIXED BINARY(31)) OPTIONS (INLINE);
   DECLARE (X, Y) FIXED BINARY (31);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END ISIGN;
DSIGN: PROCEDURE (X, Y) RETURNS (FLOAT (18)) OPTIONS (INLINE);
   DECLARE (X, Y) FLOAT (18);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END DSIGN;
   END cunk1;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cunk2: PROCEDURE (z, fnu, kode, mr, n, y, nz, tol, elim, alim)
          OPTIONS (REORDER);                                                                               
/* ***BEGIN PROLOGUE  CUNK2                                              */
/* ***REFER TO  CBESK                                                    */

/*  CUNK2 COMPUTES K(FNU,Z) AND ITS ANALYTIC CONTINUATION FROM THE RIGHT HALF */
/*  PLANE TO THE LEFT HALF PLANE BY MEANS OF THE UNIFORM ASYMPTOTIC EXPANSIONS */
/*  FOR H(KIND,FNU,ZN) AND J(FNU,ZN) WHERE ZN IS IN THE RIGHT HALF PLANE, */
/*  KIND=(3-MR)/2, MR=+1 OR -1.                                          */
/*  HERE ZN=ZR*I OR -ZR*I WHERE ZR=Z IF Z IS IN THE RIGHT HALF PLANE OR ZR=-Z */
/*  IF Z IS IN THE LEFT HALF PLANE.  MR INDICATES THE DIRECTION OF ROTATION FOR */
/*  ANALYTIC CONTINUATION.                                               */
/*  NZ=-1 MEANS AN OVERFLOW WILL OCCUR                                   */

/* ***ROUTINES CALLED  CAIRY,CS1S2,CUCHK,CUNHJ,R1MACH                    */
/* ***END PROLOGUE  CUNK2                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( mr )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( ai, arg(2), asum(2), bsum(2), cfn, ck, cs, csgn, cspn, csr(3), 
      css(3), cy(2), c1, c2, dai, phi(2), rz, s1, s2, zb, zeta1(2), zeta2(2), 
      zn, zr, phid, argd, zeta1d, zeta2d, asumd, bsumd )  COMPLEX FLOAT (18);
   DECLARE ( aarg, ang, aphi, asc, ascle, bry(3), car, cpn, c2i, c2m, c2r, 
      crsc, cscl, fmr, fn, fnf, rs1, sar, sgn, spn, x, yy )  FLOAT (18);
   DECLARE ( i, ib, iflag, ifn, il, in, inu, iuf, k, kdflg, kflag, kk, nai, 
      ndai, nw, idum, j, ipard, ic )  FIXED BINARY (31);
   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i),
             ci     STATIC INITIAL (
               0.00000000000000000E0+1.00000000000000000E0i),
             cr1    STATIC INITIAL (
               1.00000000000000000E0+1.73205080756887729E0i),
             cr2    STATIC INITIAL (
              -0.50000000000000000E0-8.66025403784438647E-01i) ) COMPLEX FLOAT (18);
   DECLARE ( hpi  STATIC INITIAL ( 1.57079632679489662E0)
           , pi   STATIC INITIAL ( 3.14159265358979324E0)
           , aic  STATIC INITIAL ( 1.26551212348464539E0) )  FLOAT (18);
   DECLARE ( cip(4)  STATIC INITIAL (
                   1.00000000000000000E0+0.00000000000000000E0i,
                   0.00000000000000000E0-1.00000000000000000E0i,
                  -1.00000000000000000E0+0.00000000000000000E0i,
                   0.00000000000000000E0+1.00000000000000000E0i) )
                COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, ib, iflag, ifn, il, in, inu, iuf, k, kdflg, kflag, kk, nai );
      PUT SKIP DATA ( ndai, nw, idum, j, ipard, ic );
      RESIGNAL;
   END;

   kdflg = 1;
   nz = 0;
/* ----------------------------------------------------------------------- */
/*     EXP(-ALIM)=EXP(-ELIM)/TOL=APPROX. ONE PRECISION GREATER THAN      */
/*     THE UNDERFLOW LIMIT                                               */
/* ----------------------------------------------------------------------- */
   cscl = 1.00000000000000000E0/tol;
   crsc = tol;
   css(1) = cscl;
   css(2) = cone;
   css(3) = crsc;
   csr(1) = crsc;
   csr(2) = cone;
   csr(3) = cscl;
   bry(1) = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
   bry(2) = 1.00000000000000000E0 / bry(1);
   bry(3) = HUGE(0.00000000000000000E0);
   x = REAL(z);
   zr = z;
   IF  x < 0 THEN
      zr = -z;
   yy = IMAG(zr);
   zn = -zr * ci;
   zb = zr;
   inu = fnu;
   fnf = fnu - inu;
   ang = -hpi * fnf;
   car = COS(ang);
   sar = SIN(ang);
   cpn = -hpi * car;
   spn = -hpi * sar;
   c2 = COMPLEX(-spn, cpn);
   kk = REM(inu,4) + 1;
   cs = cr1 * c2 * cip(kk);
   IF  yy <= 0  THEN
      DO;
         zn = CONJG(-zn);
         zb = CONJG(zb);
      END;
/* ----------------------------------------------------------------------- */
/*     K(FNU,Z) IS COMPUTED FROM H(2,FNU,-I*Z) WHERE Z IS IN THE FIRST   */
/*     QUADRANT.  FOURTH QUADRANT VALUES (YY <= 0.0) ARE COMPUTED BY     */
/*     CONJUGATION SINCE THE K FUNCTION IS REAL ON THE POSITIVE REAL AXIS */
/* ----------------------------------------------------------------------- */
   j = 2;

L05:
   DO  i = 1 TO  n;
/* ----------------------------------------------------------------------- */
/*     J FLIP FLOPS BETWEEN 1 AND 2 IN J = 3 - J                         */
/* ----------------------------------------------------------------------- */
      j = 3 - j;
      fn = fnu + (i-1);
      CALL cunhj(zn, fn, 0, tol, phi(j), arg(j), zeta1(j), zeta2(j), asum(j),
         bsum(j));
      IF  kode ^= 1  THEN
         DO;
            cfn = COMPLEX(fn, 0);
            s1 = zeta1(j) - cfn * (cfn/(zb + zeta2(j)));
         END;
      ELSE
         DO;
            s1 = zeta1(j) - zeta2(j);
         END;
/* ----------------------------------------------------------------------- */
/*     TEST FOR UNDERFLOW AND OVERFLOW                                   */
/* ----------------------------------------------------------------------- */
      rs1 = REAL(s1);
      IF  ABS(rs1) <= elim  THEN
         DO;
            IF  kdflg  = 1 THEN
               kflag = 2;
            IF  ABS(rs1) >= alim  THEN
               DO;
/* ----------------------------------------------------------------------- */
/*     REFINE  TEST AND SCALE                                            */
/* ----------------------------------------------------------------------- */
                  aphi = ABS(phi(j));
                  aarg = ABS(arg(j));
                  rs1 = rs1 + LOG(aphi) - 0.25000000000000000E0 * LOG(aarg) - aic;
                  IF  ABS(rs1) > elim THEN  GO TO L10;
                  IF  kdflg  = 1 THEN
                     kflag = 1;
                  IF  rs1 >= 0  THEN
                     DO;
                        IF  kdflg  = 1 THEN
                           kflag = 3;
                     END;
               END;
/* ----------------------------------------------------------------------- */
/*     SCALE S1 TO KEEP INTERMEDIATE ARITHMETIC ON SCALE NEAR            */
/*     EXPONENT EXTREMES                                                 */
/* ----------------------------------------------------------------------- */
            c2 = arg(j) * cr2;
            CALL cairy(c2, 0, 2, ai, nai, idum);
            CALL cairy(c2, 1, 2, dai, ndai, idum);
            s2 = cs * phi(j) * (ai*asum(j) + cr2*dai*bsum(j));
            c2r = REAL(s1);
            c2i = IMAG(s1);
            c2m = EXP(c2r) * REAL(css(kflag) );
            s1 = c2m * COMPLEX(COS(c2i), SIN(c2i) );
            s2 = s2 * s1;
            IF  kflag  = 1  THEN
               DO;
                  CALL cuchk(s2, nw, bry(1), tol);
                  IF  nw ^= 0 THEN  GO TO L10;
               END;
            IF  yy <= 0 THEN
               s2 = CONJG(s2);
            cy(kdflg) = s2;
            y(i) = s2 * csr(kflag);
            cs = -ci * cs;
            IF  kdflg  = 2 THEN  GO TO L30;
            kdflg = 2;
            ITERATE L05;
         END;

L10:
      IF  rs1 > 0 THEN  GO TO L150;
/* ----------------------------------------------------------------------- */
/*     FOR X < 0.0, THE I FUNCTION TO BE ADDED WILL OVERFLOW             */
/* ----------------------------------------------------------------------- */
      IF  x < 0 THEN  GO TO L150;
      kdflg = 1;
      y(i) = czero;
      cs = -ci * cs;
      nz = nz + 1;
      IF  i ^= 1  THEN
         DO;
            IF  y(i-1) ^= czero  THEN
               DO;
                  y(i-1) = czero;
                  nz = nz + 1;
               END;
         END;
   END;
   i = n;

L30:
   rz = 2 / zr;
   ck = fn * rz;
   ib = i + 1;
   IF  n >= ib  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     TEST LAST MEMBER FOR UNDERFLOW AND OVERFLOW, SET SEQUENCE TO ZERO */
/*     ON UNDERFLOW                                                      */
/* ----------------------------------------------------------------------- */
         fn = fnu + (n-1);
         ipard = 1;
         IF  mr ^= 0 THEN
            ipard = 0;
         CALL cunhj(zn, fn, ipard, tol, phid, argd, zeta1d, zeta2d, 
            asumd, bsumd);
         IF  kode ^= 1  THEN
            DO;
               cfn = fn;
               s1 = zeta1d - cfn * (cfn/(zb + zeta2d));
            END;
         ELSE
            DO;
               s1 = zeta1d - zeta2d;
            END;
         rs1 = REAL(s1);
         IF  ABS(rs1) <= elim  THEN
            DO;
               IF  ABS(rs1) < alim THEN  GO TO L50;
/* ----------------------------------------------------------------------- */
/*     REFINE ESTIMATE AND TEST                                          */
/* ----------------------------------------------------------------------- */
               aphi = ABS(phid);
               aarg = ABS(argd);
               rs1 = rs1 + LOG(aphi) - 0.25000000000000000E0 * LOG(aarg) - aic;
               IF  ABS(rs1) < elim THEN  GO TO L50;
            END;
         IF  rs1 > 0 THEN  GO TO L150;
/* ----------------------------------------------------------------------- */
/*     FOR X < 0.0, THE I FUNCTION TO BE ADDED WILL OVERFLOW             */
/* ----------------------------------------------------------------------- */
         IF  x < 0 THEN  GO TO L150;
         nz = n;
         y(*) = czero;
         RETURN;
/* ----------------------------------------------------------------------- */
/*     SCALED FORWARD RECURRENCE FOR REMAINDER OF THE SEQUENCE           */
/* ----------------------------------------------------------------------- */
L50:
         s1 = cy(1);
         s2 = cy(2);
         c1 = csr(kflag);
         ascle = bry(kflag);
         DO  i = ib TO  n;
            c2 = s2;
            s2 = ck * s2 + s1;
            s1 = c2;
            ck = ck + rz;
            c2 = s2 * c1;
            y(i) = c2;
            IF  kflag < 3  THEN
               DO;
                  c2r = REAL(c2);
                  c2i = IMAG(c2);
                  c2r = ABS(c2r);
                  c2i = ABS(c2i);
                  c2m = MAX(c2r,c2i);
                  IF  c2m > ascle  THEN
                     DO;
                        kflag = kflag + 1;
                        ascle = bry(kflag);
                        s1 = s1 * c1;
                        s2 = c2;
                        s1 = s1 * css(kflag);
                        s2 = s2 * css(kflag);
                        c1 = csr(kflag);
                     END;
               END;
         END;
      END;
   IF  mr  = 0 THEN
      RETURN;
/* ----------------------------------------------------------------------- */
/*     ANALYTIC CONTINUATION FOR RE(Z) < 0.0                             */
/* ----------------------------------------------------------------------- */
   nz = 0;
   fmr = mr;
   sgn = -SIGN(pi, fmr);
/* ----------------------------------------------------------------------- */
/*     CSPN AND CSGN ARE COEFF OF K AND I FUNCTIONS RESP.                */
/* ----------------------------------------------------------------------- */
   csgn = COMPLEX(0, sgn);
   IF  yy <= 0 THEN
      csgn = CONJG(csgn);
   ifn = inu + n - 1;
   ang = fnf * sgn;
   cpn = COS(ang);
   spn = SIN(ang);
   cspn = COMPLEX(cpn, spn);
   IF  REM(ifn,2)  = 1 THEN
      cspn = -cspn;
/* ----------------------------------------------------------------------- */
/*     CS=COEFF OF THE J FUNCTION TO GET THE I FUNCTION.  I(FNU,Z) IS    */
/*     COMPUTED FROM EXP(I*FNU*HPI)*J(FNU,-I*Z) WHERE Z IS IN THE FIRST  */
/*     QUADRANT.  FOURTH QUADRANT VALUES (YY <= 0.0) ARE COMPUTED BY     */
/*     CONJUGATION SINCE THE I FUNCTION IS REAL ON THE POSITIVE REAL AXIS */
/* ----------------------------------------------------------------------- */
   cs = COMPLEX(car, -sar) * csgn;
   in = REM(ifn,4) + 1;
   c2 = cip(in);
   cs = cs * CONJG(c2);
   asc = bry(1);
   kk = n;
   kdflg = 1;
   ib = ib - 1;
   ic = ib - 1;
   iuf = 0;

L65:
   DO  k = 1 TO  n;
/* ----------------------------------------------------------------------- */
/*     LOGIC TO SORT OUT CASES WHOSE PARAMETERS WERE SET FOR THE K       */
/*     FUNCTION ABOVE                                                    */
/* ----------------------------------------------------------------------- */
      fn = fnu + (kk-1);
      IF  n > 2 THEN  GO TO L80;

L70:
      phid = phi(j);
      argd = arg(j);
      zeta1d = zeta1(j);
      zeta2d = zeta2(j);
      asumd = asum(j);
      bsumd = bsum(j);
      j = 3 - j;
      GO TO L90;

L80:
      IF  ^ (kk  = n   &   ib < n)  THEN
         DO;
            IF  kk  = ib  |   kk  = ic THEN  GO TO L70;
            CALL cunhj(zn, fn, 0, tol, phid, argd, zeta1d, zeta2d, 
               asumd, bsumd);
         END;

L90:
      IF  kode ^= 1  THEN
         DO;
            cfn = fn;
            s1 = -zeta1d + cfn * (cfn/(zb + zeta2d));
         END;
      ELSE
         DO;
            s1 = -zeta1d + zeta2d;
         END;
/* ----------------------------------------------------------------------- */
/*     TEST FOR UNDERFLOW AND OVERFLOW                                   */
/* ----------------------------------------------------------------------- */
      rs1 = REAL(s1);
      IF  ABS(rs1) > elim THEN  GO TO L110;
      IF  kdflg  = 1 THEN
         iflag = 2;
      IF  ABS(rs1) >= alim  THEN
         DO;
/* ----------------------------------------------------------------------- */
/*     REFINE  TEST AND SCALE                                            */
/* ----------------------------------------------------------------------- */
            aphi = ABS(phid);
            aarg = ABS(argd);
            rs1 = rs1 + LOG(aphi) - 0.25000000000000000E0 * LOG(aarg) - aic;
            IF  ABS(rs1) > elim THEN  GO TO L110;
            IF  kdflg  = 1 THEN
               iflag = 1;
            IF  rs1 >= 0  THEN
               DO;
                  IF  kdflg  = 1 THEN
                     iflag = 3;
               END;
         END;
      CALL cairy(argd, 0, 2, ai, nai, idum);
      CALL cairy(argd, 1, 2, dai, ndai, idum);
      s2 = cs * phid * (ai*asumd + dai*bsumd);
      c2r = REAL(s1);
      c2i = IMAG(s1);
      c2m = EXP(c2r) * REAL(css(iflag) );
      s1 = c2m * COMPLEX(COS(c2i), SIN(c2i) );
      s2 = s2 * s1;
      IF  iflag  = 1  THEN
         DO;
            CALL cuchk(s2, nw, bry(1), tol);
            IF  nw ^= 0 THEN
               s2 = 0;
         END;

L100:
      IF  yy <= 0 THEN
         s2 = CONJG(s2);
      cy(kdflg) = s2;
      c2 = s2;
      s2 = s2 * csr(iflag);
/* ----------------------------------------------------------------------- */
/*     ADD I AND K FUNCTIONS, K SEQUENCE IN Y(I), I=1,N                  */
/* ----------------------------------------------------------------------- */
      s1 = y(kk);
      IF  kode ^= 1  THEN
         DO;
            CALL cs1s2(zr, s1, s2, nw, asc, alim, iuf);
            nz = nz + nw;
         END;
      y(kk) = s1 * cspn + s2;
      kk = kk - 1;
      cspn = -cspn;
      cs = -cs * ci;
      IF  c2  = czero  THEN
         DO;
            kdflg = 1;
            ITERATE L65;
         END;
      IF  kdflg  = 2 THEN  GO TO L130;
      kdflg = 2;
      ITERATE;

L110:
      IF  rs1 > 0 THEN  GO TO L150;
      s2 = czero;
      GO TO L100;
   END;
   k = n;

L130:
   il = n - k;
   IF  il  = 0 THEN
      RETURN;
/* ----------------------------------------------------------------------- */
/*     RECUR BACKWARD FOR REMAINDER OF I SEQUENCE AND ADD IN THE         */
/*     K FUNCTIONS, SCALING THE I SEQUENCE DURING RECURRENCE TO KEEP     */
/*     INTERMEDIATE ARITHMETIC ON SCALE NEAR EXPONENT EXTREMES.          */
/* ----------------------------------------------------------------------- */
   s1 = cy(1);
   s2 = cy(2);
   cs = csr(iflag);
   ascle = bry(iflag);
   fn = inu + il;
   DO  i = 1 TO  il;
      c2 = s2;
      s2 = s1 + COMPLEX(fn+fnf, 0) * rz * s2;
      s1 = c2;
      fn = fn - 1.00000000000000000E0;
      c2 = s2 * cs;
      ck = c2;
      c1 = y(kk);
      IF  kode ^= 1  THEN
         DO;
            CALL cs1s2(zr, c1, c2, nw, asc, alim, iuf);
            nz = nz + nw;
         END;
      y(kk) = c1 * cspn + c2;
      kk = kk - 1;
      cspn = -cspn;
      IF  iflag < 3  THEN
         DO;
            c2r = REAL(ck);
            c2i = IMAG(ck);
            c2r = ABS(c2r);
            c2i = ABS(c2i);
            c2m = MAX(c2r, c2i);
            IF  c2m > ascle  THEN
               DO;
                  iflag = iflag + 1;
                  ascle = bry(iflag);
                  s1 = s1 * cs;
                  s2 = ck;
                  s1 = s1 * css(iflag);
                  s2 = s2 * css(iflag);
                  cs = csr(iflag);
               END;
         END;
   END;
   RETURN;

L150:
   nz = -1;
   RETURN;

DECLARE SIGN GENERIC (DSIGN WHEN (FLOAT, FLOAT),
   ISIGN WHEN (FIXED BINARY, FIXED BINARY) );
ISIGN: PROCEDURE (X, Y) RETURNS (FIXED BINARY(31)) OPTIONS (INLINE);
   DECLARE (X, Y) FIXED BINARY (31);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END ISIGN;
DSIGN: PROCEDURE (X, Y) RETURNS (FLOAT (18)) OPTIONS (INLINE);
   DECLARE (X, Y) FLOAT (18);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END DSIGN;
   END cunk2;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbuni: PROCEDURE (z, fnu, kode, n, y, nz, nui, nlast, fnul, tol, elim, alim)
         OPTIONS (REORDER);

/* ***BEGIN PROLOGUE  CBUNI                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*   CBUNI COMPUTES THE I BESSEL FUNCTION FOR LARGE ABS(Z) > FNUL AND    */
/*   FNU+N-1 < FNUL.  THE ORDER IS INCREASED FROM FNU+N-1 GREATER THAN FNUL */
/*   BY ADDING NUI AND COMPUTING ACCORDING TO THE UNIFORM ASYMPTOTIC EXPANSION */
/*   FOR I(FNU,Z) ON IFORM=1 AND THE EXPANSION FOR J(FNU,Z) ON IFORM=2   */

/* ***ROUTINES CALLED  CUNI1,CUNI2,R1MACH                                */
/* ***END PROLOGUE  CBUNI                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( nui )  FIXED BINARY (31);
   DECLARE ( nlast )  FIXED BINARY (31);
   DECLARE ( fnul )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( cscl, cscr, cy(2), rz, st, s1, s2 )  COMPLEX FLOAT (18);
   DECLARE ( ax, ay, dfnu, fnui, gnu, xx, yy, ascle, bry(3), str, sti, stm ) 
             FLOAT (18);
   DECLARE ( i, iflag, iform, k, nl, nw )  FIXED BINARY (31);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, iflag, iform, k, nl, nw );
      RESIGNAL;
   END;

   nz = 0;
   xx = REAL(z);
   yy = IMAG(z);
   ax = ABS(xx) * 1.73205080756887000E0;
   ay = ABS(yy);
   iform = 1;
   IF  ay > ax THEN
      iform = 2;
   IF  nui  = 0 THEN  GO TO L40;
   fnui = nui;
   dfnu = fnu + (n-1);
   gnu = dfnu + fnui;
   IF  iform ^= 2  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     ASYMPTOTIC EXPANSION FOR I(FNU,Z) FOR LARGE FNU APPLIED IN        */
/*     -PI/3 <= ARG(Z) <= PI/3                                           */
/* ----------------------------------------------------------------------- */
         CALL cuni1(z, gnu, kode, 2, cy, nw, nlast, fnul, tol, elim, alim);
      END;
   ELSE
      DO;
/* ----------------------------------------------------------------------- */
/*     ASYMPTOTIC EXPANSION FOR J(FNU,Z*EXP(M*HPI)) FOR LARGE FNU APPLIED */
/*     IN PI/3 < ABS(ARG(Z)) <= PI/2 WHERE M=+I OR -I AND HPI=PI/2       */
/* ----------------------------------------------------------------------- */
         CALL cuni2(z, gnu, kode, 2, cy, nw, nlast, fnul, tol, elim, alim);
      END;
   IF  nw >= 0  THEN
      DO;
         IF  nw ^= 0 THEN  GO TO L50;
         ay = ABS(cy(1));
/* ---------------------------------------------------------------------- */
/*     SCALE BACKWARD RECURRENCE, BRY(3) IS DEFINED BUT NEVER USED       */
/* ---------------------------------------------------------------------- */
         bry(1) = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
         bry(2) = 1.00000000000000000E0 / bry(1);
         bry(3) = bry(2);
         iflag = 2;
         ascle = bry(2);
         ax = 1.00000000000000000E0;
         cscl = ax;
         IF  ay <= bry(1)  THEN
            DO;
               iflag = 1;
               ascle = bry(1);
               ax = 1.00000000000000000E0 / tol;
               cscl = ax;
            END;
         ELSE
            DO;
               IF  ay >= bry(2)  THEN
                  DO;
                     iflag = 3;
                     ascle = bry(3);
                     ax = tol;
                     cscl = ax;
                  END;
            END;
         ay = 1.00000000000000000E0 / ax;
         cscr = ay;
         s1 = cy(2) * cscl;
         s2 = cy(1) * cscl;
         rz = 2.00000000000000000E0 / z;
         DO  i = 1 TO  nui;
            st = s2;
            s2 = COMPLEX(dfnu+fnui, 0) * rz * s2 + s1;
            s1 = st;
            fnui = fnui - 1.00000000000000000E0;
            IF  iflag < 3  THEN
               DO;
                  st = s2 * cscr;
                  str = REAL(st);
                  sti = IMAG(st);
                  str = ABS(str);
                  sti = ABS(sti);
                  stm = MAX(str,sti);
                  IF  stm > ascle  THEN
                     DO;
                        iflag = iflag + 1;
                        ascle = bry(iflag);
                        s1 = s1 * cscr;
                        s2 = st;
                        ax = ax * tol;
                        ay = 1.00000000000000000E0 / ax;
                        cscl = ax;
                        cscr = ay;
                        s1 = s1 * cscl;
                        s2 = s2 * cscl;
                     END;
               END;
         END;
         y(n) = s2 * cscr;
         IF  n  = 1 THEN
            RETURN;
         nl = n - 1;
         fnui = nl;
         k = nl;
         DO  i = 1 TO  nl;
            st = s2;
            s2 = COMPLEX(fnu+fnui, 0) * rz * s2 + s1;
            s1 = st;
            st = s2 * cscr;
            y(k) = st;
            fnui = fnui - 1.00000000000000000E0;
            k = k - 1;
            IF  iflag < 3  THEN
               DO;
                  str = REAL(st);
                  sti =  IMAG(st);
                  str = ABS(str);
                  sti = ABS(sti);
                  stm = MAX(str,sti);
                  IF  stm > ascle  THEN
                     DO;
                        iflag = iflag + 1;
                        ascle = bry(iflag);
                        s1 = s1 * cscr;
                        s2 = st;
                        ax = ax * tol;
                        ay = 1.00000000000000000E0 / ax;
                        cscl = ax;
                        cscr = ay;
                        s1 = s1 * cscl;
                        s2 = s2 * cscl;
                     END;
               END;
         END;
         RETURN;
      END;

L30:
   nz = -1;
   IF  nw  = -2 THEN
      nz = -2;
   RETURN;

L40:
   IF  iform ^= 2  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     ASYMPTOTIC EXPANSION FOR I(FNU,Z) FOR LARGE FNU APPLIED IN        */
/*     -PI/3 <= ARG(Z) <= PI/3                                           */
/* ----------------------------------------------------------------------- */
         CALL cuni1(z, fnu, kode, n, y, nw, nlast, fnul, tol, elim, alim);
      END;
   ELSE
      DO;
/* ----------------------------------------------------------------------- */
/*     ASYMPTOTIC EXPANSION FOR J(FNU,Z*EXP(M*HPI)) FOR LARGE FNU APPLIED */
/*     IN PI/3 < ABS(ARG(Z)) <= PI/2 WHERE M=+I OR -I AND HPI=PI/2       */
/* ----------------------------------------------------------------------- */
         CALL cuni2(z, fnu, kode, n, y, nw, nlast, fnul, tol, elim, alim);
      END;
   IF  nw < 0 THEN  GO TO L30;
   nz = nw;
   RETURN;

L50:
   nlast = n;
   RETURN;
   END cbuni;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cuni1: PROCEDURE (z, fnu, kode, n, y, nz, nlast, fnul, tol, elim, alim)
          OPTIONS (REORDER);                                                                      
/* ***BEGIN PROLOGUE  CUNI1                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*     CUNI1 COMPUTES I(FNU,Z)  BY MEANS OF THE UNIFORM ASYMPTOTIC       */
/*     EXPANSION FOR I(FNU,Z) IN -PI/3 <= ARG Z <= PI/3.                 */

/*     FNUL IS THE SMALLEST ORDER PERMITTED FOR THE ASYMPTOTIC           */
/*     EXPANSION. NLAST=0 MEANS ALL OF THE Y VALUES WERE SET.            */
/*     NLAST.NE.0 IS THE NUMBER LEFT TO BE COMPUTED BY ANOTHER           */
/*     FORMULA FOR ORDERS FNU TO FNU+NLAST-1 BECAUSE FNU+NLAST-1 < FNUL. */
/*     Y(I)=CZERO FOR I=NLAST+1,N                                        */

/* ***ROUTINES CALLED  CUCHK,CUNIK,CUOIK,R1MACH                          */
/* ***END PROLOGUE  CUNI1                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( nlast )  FIXED BINARY (31);
   DECLARE ( fnul )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( cfn, crsc, cscl, csr(3), css(3), c1, c2, cwrk(16), phi, rz, sum, 
      s1, s2, zeta1, zeta2, cy(2) )  COMPLEX FLOAT (18);
   DECLARE ( aphi, ascle, bry(3), c2i, c2m, c2r, fn, rs1, yy )  FLOAT (18);
   DECLARE ( i, iflag, init, k, m, nd, nn, nuf, nw )  FIXED BINARY (31);
   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i) ) COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, iflag, init, k, m, nd, nn, nuf, nw );
      RESIGNAL;
   END;

   nz = 0;
   nd = n;
   nlast = 0;
/* ----------------------------------------------------------------------- */
/*     COMPUTED VALUES WITH EXPONENTS BETWEEN ALIM AND ELIM IN MAGNITUDE */
/*     ARE SCALED TO KEEP INTERMEDIATE ARITHMETIC ON SCALE,              */
/*     EXP(ALIM) = EXP(ELIM)*TOL                                         */
/* ----------------------------------------------------------------------- */
   cscl = COMPLEX(1.00000000000000000E0/tol, 0);
   crsc = tol;
   css(1) = cscl;
   css(2) = cone;
   css(3) = crsc;
   csr(1) = crsc;
   csr(2) = cone;
   csr(3) = cscl;
   bry(1) = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
/* ----------------------------------------------------------------------- */
/*     CHECK FOR UNDERFLOW AND OVERFLOW ON FIRST MEMBER                  */
/* ----------------------------------------------------------------------- */
   fn = MAX(fnu, 1.00000000000000000E0);
   init = 0;
   CALL cunik(z, fn, 1, 1, tol, init, phi, zeta1, zeta2, sum, cwrk);
   IF  kode ^= 1  THEN
      DO;
         cfn = fn;
         s1 = -zeta1 + cfn * (cfn/(z + zeta2));
      END;
   ELSE
      DO;
         s1 = -zeta1 + zeta2;
      END;
   rs1 = REAL(s1);
   IF  ABS(rs1) > elim THEN  GO TO L70;

L10:
   nn = MIN(2,nd);
   DO  i = 1 TO  nn;
      fn = fnu + (nd-i);
      init = 0;
      CALL cunik(z, fn, 1, 0, tol, init, phi, zeta1, zeta2, sum, cwrk);
      IF  kode ^= 1  THEN
         DO;
            cfn = fn;
            yy =  IMAG(z);
            s1 = -zeta1 + cfn * (cfn/(z+zeta2)) + COMPLEX(0, yy);
         END;
      ELSE
         DO;
            s1 = -zeta1 + zeta2;
         END;
/* ----------------------------------------------------------------------- */
/*     TEST FOR UNDERFLOW AND OVERFLOW                                   */
/* ----------------------------------------------------------------------- */
      rs1 = REAL(s1);
      IF  ABS(rs1) > elim THEN  GO TO L50;
      IF  i  = 1 THEN
         iflag = 2;
      IF  ABS(rs1) >= alim  THEN
         DO;
/* ----------------------------------------------------------------------- */
/*     REFINE  TEST AND SCALE                                            */
/* ----------------------------------------------------------------------- */
            aphi = ABS(phi);
            rs1 = rs1 + LOG(aphi);
            IF  ABS(rs1) > elim THEN  GO TO L50;
            IF  i  = 1 THEN
               iflag = 1;
            IF  rs1 >= 0  THEN
               DO;
                  IF  i  = 1 THEN
                     iflag = 3;
               END;
         END;
/* ----------------------------------------------------------------------- */
/*     SCALE S1 IF ABS(S1) < ASCLE                                       */
/* ----------------------------------------------------------------------- */
      s2 = phi * sum;
      c2r = REAL(s1);
      c2i = IMAG(s1);
      c2m = EXP(c2r) * REAL(css(iflag) );
      s1 = c2m * COMPLEX(COS(c2i), SIN(c2i) );
      s2 = s2 * s1;
      IF  iflag  = 1  THEN
         DO;
            CALL cuchk(s2, nw, bry(1), tol);
            IF  nw ^= 0 THEN  GO TO L50;
         END;
      m = nd - i + 1;
      cy(i) = s2;
      y(m) = s2 * csr(iflag);
   END;
   IF  nd > 2  THEN
      DO;
         rz = 2.00000000000000000E0 / z;
         bry(2) = 1.00000000000000000E0 / bry(1);
         bry(3) = HUGE(0.00000000000000000E0);
         s1 = cy(1);
         s2 = cy(2);
         c1 = csr(iflag);
         ascle = bry(iflag);
         k = nd - 2;
         fn = k;
         DO  i = 3 TO  nd;
            c2 = s2;
            s2 = s1 + COMPLEX(fnu+fn, 0) * rz * s2;
            s1 = c2;
            c2 = s2 * c1;
            y(k) = c2;
            k = k - 1;
            fn = fn - 1.00000000000000000E0;
            IF  iflag < 3  THEN
               DO;
                  c2r = REAL(c2);
                  c2i = IMAG(c2);
                  c2r = ABS(c2r);
                  c2i = ABS(c2i);
                  c2m = MAX(c2r,c2i);
                  IF  c2m > ascle  THEN
                     DO;
                        iflag = iflag + 1;
                        ascle = bry(iflag);
                        s1 = s1 * c1;
                        s2 = c2;
                        s1 = s1 * css(iflag);
                        s2 = s2 * css(iflag);
                        c1 = csr(iflag);
                     END;
               END;
         END;
      END;

L40:
   RETURN;
/* ----------------------------------------------------------------------- */
/*     SET UNDERFLOW AND UPDATE PARAMETERS                               */
/* ----------------------------------------------------------------------- */
L50:
   IF  rs1 <= 0  THEN
      DO;
         y(nd) = czero;
         nz = nz + 1;
         nd = nd - 1;
         IF  nd  = 0 THEN  GO TO L40;
         CALL cuoik(z, fnu, kode, 1, nd, y, nuf, tol, elim, alim);
         IF  nuf >= 0  THEN
            DO;
               nd = nd - nuf;
               nz = nz + nuf;
               IF  nd  = 0 THEN  GO TO L40;
               fn = fnu + (nd-1);
               IF  fn >= fnul THEN  GO TO L10;
               nlast = nd;
               RETURN;
            END;
      END;

L60:
   nz = -1;
   RETURN;

L70:
   IF  rs1 > 0 THEN  GO TO L60;
   nz = n;
   y(*) = czero;
   RETURN;
   END cuni1;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cuni2: PROCEDURE (z, fnu, kode, n, y, nz, nlast, fnul, tol, elim, alim) OPTIONS 
(REORDER);                                                                      
/* ***BEGIN PROLOGUE  CUNI2                                              */
/* ***REFER TO  CBESI,CBESK                                              */

/*     CUNI2 COMPUTES I(FNU,Z) IN THE RIGHT HALF PLANE BY MEANS OF       */
/*     UNIFORM ASYMPTOTIC EXPANSION FOR J(FNU,ZN) WHERE ZN IS Z*I        */
/*     OR -Z*I AND ZN IS IN THE RIGHT HALF PLANE ALSO.                   */

/*     FNUL IS THE SMALLEST ORDER PERMITTED FOR THE ASYMPTOTIC           */
/*     EXPANSION.  NLAST=0 MEANS ALL OF THE Y VALUES WERE SET.           */
/*     NLAST.NE.0 IS THE NUMBER LEFT TO BE COMPUTED BY ANOTHER           */
/*     FORMULA FOR ORDERS FNU TO FNU+NLAST-1 BECAUSE FNU+NLAST-1 < FNUL. */
/*     Y(I) = CZERO FOR I=NLAST+1,N                                      */

/* ***ROUTINES CALLED  CAIRY,CUCHK,CUNHJ,CUOIK,R1MACH                    */
/* ***END PROLOGUE  CUNI2                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( nlast )  FIXED BINARY (31);
   DECLARE ( fnul )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( ai, arg, asum, bsum, cfn, cid, crsc, cscl, csr(3), css(3), cy(2), 
      c1, c2, dai, phi, rz, s1, s2, zb, zeta1, zeta2, zn, zar )  COMPLEX FLOAT (18);
   DECLARE ( aarg, ang, aphi, ascle, ay, bry(3), car, c2i, c2m, c2r, fn, rs1, 
      sar, yy )  FLOAT (18);
   DECLARE ( i, iflag, in, inu, j, k, nai, nd, ndai, nn, nuf, nw, idum )  
      FIXED BINARY (31);
   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i),
             ci     STATIC INITIAL (
               0.00000000000000000E0+1.00000000000000000E0i) ) COMPLEX FLOAT (18);
   DECLARE ( cip(4)  STATIC INITIAL (
                   1.00000000000000000E0+0.00000000000000000E0i,
                   0.00000000000000000E0+1.00000000000000000E0i,
                  -1.00000000000000000E0+0.00000000000000000E0i,
                   0.00000000000000000E0-1.00000000000000000E0i) )
                COMPLEX FLOAT (18);
   DECLARE ( hpi  STATIC INITIAL ( 1.57079632679489662E0),
             aic  STATIC INITIAL ( 1.265512123484645396E0)
               )  FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, iflag, in, inu, j, k, nai, nd, ndai, nn, nuf, nw, idum );
      RESIGNAL;
   END;

   nz = 0;
   nd = n;
   nlast = 0;
/* ----------------------------------------------------------------------- */
/*     COMPUTED VALUES WITH EXPONENTS BETWEEN ALIM AND ELIM IN MAGNITUDE */
/*     ARE SCALED TO KEEP INTERMEDIATE ARITHMETIC ON SCALE,              */
/*     EXP(ALIM) = EXP(ELIM)*TOL                                         */
/* ----------------------------------------------------------------------- */
   cscl = COMPLEX(1.00000000000000000E0/tol, 0);
   crsc = tol;
   css(1) = cscl;
   css(2) = cone;
   css(3) = crsc;
   csr(1) = crsc;
   csr(2) = cone;
   csr(3) = cscl;
   bry(1) = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
   yy =  IMAG(z);
/* ----------------------------------------------------------------------- */
/*     ZN IS IN THE RIGHT HALF PLANE AFTER ROTATION BY CI OR -CI         */
/* ----------------------------------------------------------------------- */
   zn = -z * ci;
   zb = z;
   cid = -ci;
   inu = fnu;
   ang = hpi * (fnu - inu);
   car = COS(ang);
   sar = SIN(ang);
   c2 = COMPLEX(car, sar);
   zar = c2;
   in = inu + n - 1;
   in = REM(in,4);
   c2 = c2 * cip(in+1);
   IF  yy <= 0  THEN
      DO;
         zn = CONJG(-zn);
         zb = CONJG(zb);
         cid = -cid;
         c2 = CONJG(c2);
      END;
/* ----------------------------------------------------------------------- */
/*     CHECK FOR UNDERFLOW AND OVERFLOW ON FIRST MEMBER                  */
/* ----------------------------------------------------------------------- */
   fn = MAX(fnu,1.00000000000000000E0);
   CALL cunhj(zn, fn, 1, tol, phi, arg, zeta1, zeta2, asum, bsum);
   IF  kode ^= 1  THEN
      DO;
         cfn = fnu;
         s1 = -zeta1 + cfn * (cfn/(zb + zeta2));
      END;
   ELSE
      DO;
         s1 = -zeta1 + zeta2;
      END;
   rs1 = REAL(s1);
   IF  ABS(rs1) > elim THEN  GO TO L70;

L10:
   nn = MIN(2,nd);
   DO  i = 1 TO  nn;
      fn = fnu + (nd-i);
      CALL cunhj(zn, fn, 0, tol, phi, arg, zeta1, zeta2, asum, bsum);
      IF  kode ^= 1  THEN
         DO;
            cfn = fn;
            ay = ABS(yy);
            s1 = -zeta1 + cfn * (cfn/(zb+zeta2)) + COMPLEX(0, ay);
         END;
      ELSE
         DO;
            s1 = -zeta1 + zeta2;
         END;
/* ----------------------------------------------------------------------- */
/*     TEST FOR UNDERFLOW AND OVERFLOW                                   */
/* ----------------------------------------------------------------------- */
      rs1 = REAL(s1);
      IF  ABS(rs1) > elim THEN  GO TO L50;
      IF  i  = 1 THEN
         iflag = 2;
      IF  ABS(rs1) >= alim  THEN
         DO;
/* ----------------------------------------------------------------------- */
/*     REFINE  TEST AND SCALE                                            */
/* ----------------------------------------------------------------------- */
            aphi = ABS(phi);
            aarg = ABS(arg);
            rs1 = rs1 + LOG(aphi) - 0.25000000000000000E0 * LOG(aarg) - aic;
            IF  ABS(rs1) > elim THEN  GO TO L50;
            IF  i  = 1 THEN
               iflag = 1;
            IF  rs1 >= 0  THEN
               DO;
                  IF  i  = 1 THEN
                     iflag = 3;
               END;
         END;
/* ----------------------------------------------------------------------- */
/*     SCALE S1 TO KEEP INTERMEDIATE ARITHMETIC ON SCALE NEAR            */
/*     EXPONENT EXTREMES                                                 */
/* ----------------------------------------------------------------------- */
      CALL cairy(arg, 0, 2, ai, nai, idum);
      CALL cairy(arg, 1, 2, dai, ndai, idum);
      s2 = phi * (ai*asum + dai*bsum);
      c2r = REAL(s1);
      c2i = IMAG(s1);
      c2m = EXP(c2r) * REAL(css(iflag) );
      s1 = c2m * COMPLEX(COS(c2i), SIN(c2i) );
      s2 = s2 * s1;
      IF  iflag  = 1  THEN
         DO;
            CALL cuchk(s2, nw, bry(1), tol);
            IF  nw ^= 0 THEN  GO TO L50;
         END;
      IF  yy <= 0 THEN
         s2 = CONJG(s2);
      j = nd - i + 1;
      s2 = s2 * c2;
      cy(i) = s2;
      y(j) = s2 * csr(iflag);
      c2 = c2 * cid;
   END;
   IF  nd > 2  THEN
      DO;
         rz = 2.00000000000000000E0 / z;
         bry(2) = 1.00000000000000000E0 / bry(1);
         bry(3) = HUGE(0.00000000000000000E0);
         s1 = cy(1);
         s2 = cy(2);
         c1 = csr(iflag);
         ascle = bry(iflag);
         k = nd - 2;
         fn = k;
         DO  i = 3 TO  nd;
            c2 = s2;
            s2 = s1 + (fnu + fn) * rz * s2;
            s1 = c2;
            c2 = s2 * c1;
            y(k) = c2;
            k = k - 1;
            fn = fn - 1.00000000000000000E0;
            IF  iflag < 3  THEN
               DO;
                  c2r = REAL(c2);
                  c2i = IMAG(c2);
                  c2r = ABS(c2r);
                  c2i = ABS(c2i);
                  c2m = MAX(c2r,c2i);
                  IF  c2m > ascle  THEN
                     DO;
                        iflag = iflag + 1;
                        ascle = bry(iflag);
                        s1 = s1 * c1;
                        s2 = c2;
                        s1 = s1 * css(iflag);
                        s2 = s2 * css(iflag);
                        c1 = csr(iflag);
                     END;
               END;
         END;
      END;

L40:
   RETURN;

L50:
   IF  rs1 <= 0  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     SET UNDERFLOW AND UPDATE PARAMETERS                               */
/* ----------------------------------------------------------------------- */
         y(nd) = czero;
         nz = nz + 1;
         nd = nd - 1;
         IF  nd  = 0 THEN  GO TO L40;
         CALL cuoik(z, fnu, kode, 1, nd, y, nuf, tol, elim, alim);
         IF  nuf >= 0  THEN
            DO;
               nd = nd - nuf;
               nz = nz + nuf;
               IF  nd  = 0 THEN  GO TO L40;
               fn = fnu + (nd-1);
               IF  fn >= fnul  THEN
                  DO;
/*      FN = AIMAG(CID)                                                  */
/*      J = NUF + 1                                                      */
/*      K = REM(J,4) + 1                                                 */
/*      S1 = CIP(K)                                                      */
/*      IF (FN < 0.0) S1 = CONJG(S1)                                     */
/*      C2 = C2*S1                                                       */
                     in = inu + nd - 1;
                     in = REM(in,4) + 1;
                     c2 = zar * cip(in);
                     IF  yy <= 0 THEN
                        c2 = CONJG(c2);
                     GO TO L10;
                  END;
               nlast = nd;
               RETURN;
            END;
      END;

L60:
   nz = -1;
   RETURN;

L70:
   IF  rs1 > 0 THEN  GO TO L60;
   nz = n;
   y(*) = czero;
   RETURN;
   END cuni2;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cs1s2: PROCEDURE (zr, s1, s2, nz, ascle, alim, iuf) OPTIONS (REORDER);          
/* ***BEGIN PROLOGUE  CS1S2                                              */
/* ***REFER TO  CBESK,CAIRY                                              */

/*     CS1S2 TESTS FOR A POSSIBLE UNDERFLOW RESULTING FROM THE ADDITION OF */
/*     THE I AND K FUNCTIONS IN THE ANALYTIC CONTINUATION FORMULA WHERE S1=K */
/*     FUNCTION AND S2=I FUNCTION.                                       */
/*     ON KODE=1 THE I AND K FUNCTIONS ARE DIFFERENT ORDERS OF MAGNITUDE, */
/*     BUT FOR KODE=2 THEY CAN BE OF THE SAME ORDER OF MAGNITUDE AND THE */
/*     MAXIMUM MUST BE AT LEAST ONE PRECISION ABOVE THE UNDERFLOW LIMIT. */

/* ***ROUTINES CALLED  (NONE)                                            */
/* ***END PROLOGUE  CS1S2                                                */

   DECLARE ( zr )  COMPLEX FLOAT (18);
   DECLARE ( s1 )  COMPLEX FLOAT (18);
   DECLARE ( s2 )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( ascle )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);
   DECLARE ( iuf )  FIXED BINARY (31);

   DECLARE ( c1, s1d )  COMPLEX FLOAT (18);
   DECLARE ( aa, aln, as1, as2, xx )  FLOAT (18);

   DECLARE czero  STATIC INITIAL ( 0+0i )  COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( nz, iuf );
      RESIGNAL;
   END;

   nz  = 0;
   as1 = ABS(s1);
   as2 = ABS(s2);
   aa  = REAL(s1);
   aln = IMAG(s1);
   IF  aa ^= 0  |   aln ^= 0  THEN
      DO;
         IF  as1 ^= 0  THEN
            DO;
               xx = REAL(zr);
               aln = -xx - xx + LOG(as1);
               s1d = s1;
               s1 = czero;
               as1 = 0;
               IF  aln >= -alim  THEN
                  DO;
                     c1 = LOG(s1d) - zr - zr;
                     s1 = EXP(c1);
                     as1 = ABS(s1);
                     iuf = iuf + 1;
                  END;
            END;
      END;
   aa = MAX(as1,as2);
   IF  aa > ascle THEN
      RETURN;
   s1 = czero;
   s2 = czero;
   nz = 1;
   iuf = 0;
   RETURN;
   END cs1s2;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cshch: PROCEDURE (z, csh, cch) OPTIONS (REORDER);                               
/* ***BEGIN PROLOGUE  CSHCH                                              */
/* ***REFER TO  CBESK,CBESH                                              */

/*     CSHCH COMPUTES THE COMPLEX HYPERBOLIC FUNCTIONS CSH=SINH(X+I*Y)   */
/*     AND CCH=COSH(X+I*Y), WHERE I**2=-1.                               */

/* ***ROUTINES CALLED  (NONE)                                            */
/* ***END PROLOGUE  CSHCH                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( csh )  COMPLEX FLOAT (18);
   DECLARE ( cch )  COMPLEX FLOAT (18);

   DECLARE ( cchi, cchr, ch, cn, cshi, cshr, sh, sn, x, y )  FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( x, y );
      RESIGNAL;
   END;

   x  = REAL(z);
   y  = IMAG(z);
   sh = SINH(x);
   ch = COSH(x);
   sn = SIN(y);
   cn = COS(y);
   cshr = sh * cn;
   cshi = ch * sn;
   csh = COMPLEX(cshr, cshi);
   cchr = ch * cn;
   cchi = sh * sn;
   cch = COMPLEX(cchr, cchi);
   RETURN;
   END cshch;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
crati: PROCEDURE (z, fnu, n, cy, tol) OPTIONS (REORDER);                        
/* ***BEGIN PROLOGUE  CRATI                                              */
/* ***REFER TO  CBESI,CBESK,CBESH                                        */

/*   CRATI COMPUTES RATIOS OF I BESSEL FUNCTIONS BY BACKWARD RECURRENCE. */
/*   THE STARTING INDEX IS DETERMINED BY FORWARD RECURRENCE AS DESCRIBED IN */
/*   J. RES. OF NAT. BUR. OF STANDARDS-B, MATHEMATICAL SCIENCES, VOL 77B, */
/*   P111-114, SEPTEMBER 1973, BESSEL FUNCTIONS I AND J OF COMPLEX ARGUMENT */
/*   AND INTEGER ORDER, BY D. J. SOOKNE.                                 */

/* ***ROUTINES CALLED  (NONE)                                            */
/* ***END PROLOGUE  CRATI                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( cy(n) )  COMPLEX FLOAT (18);
   DECLARE ( tol )  FLOAT (18);

   DECLARE ( cdfnu, pt, p1, p2, rz, t1 )  COMPLEX FLOAT (18);
   DECLARE ( ak, amagz, ap1, ap2, arg, az, dfnu, fdnu, flam, fnup, rap1, rho, 
      test, test1 )  FLOAT (18);
   DECLARE ( i, id, idnu, inu, itime, k, kk, magz )  FIXED BINARY (31);

   DECLARE ( czero  STATIC INITIAL (
               0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL (
               1.00000000000000000E0+0.00000000000000000E0i) ) COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, id, idnu, inu, itime, k, kk, magz );
      RESIGNAL;
   END;

   az = ABS(z);
   inu = fnu;
   idnu = inu + n - 1;
   fdnu = idnu;
   magz = az;
   amagz = magz + 1;
   fnup = MAX(amagz, fdnu);
   id = idnu - magz - 1;
   itime = 1;
   k = 1;
   rz = (cone+cone) / z;
   t1 = fnup * rz;
   p2 = -t1;
   p1 = cone;
   t1 = t1 + rz;
   IF  id > 0 THEN id = 0;
   ap2 = ABS(p2);
   ap1 = ABS(p1);
/* ----------------------------------------------------------------------- */
/*     THE OVERFLOW TEST ON K(FNU+I-1,Z) BEFORE THE CALL TO CBKNX        */
/*     GUARANTEES THAT P2 IS ON SCALE. SCALE TEST1 AND ALL SUBSEQUENT P2 */
/*     VALUES BY AP1 TO ENSURE THAT AN OVERFLOW DOES NOT OCCUR PREMATURELY. */
/* ----------------------------------------------------------------------- */
   arg = (ap2+ap2) / (ap1*tol);
   test1 = SQRT(arg);
   test = test1;
   rap1 = 1.00000000000000000E0 / ap1;
   p1 = p1 * rap1;
   p2 = p2 * rap1;
   ap2 = ap2 * rap1;

L10:
   k = k + 1;
   ap1 = ap2;
   pt = p2;
   p2 = p1 - t1 * p2;
   p1 = pt;
   t1 = t1 + rz;
   ap2 = ABS(p2);
   IF  ap1 <= test THEN  GO TO L10;
   IF  itime ^= 2  THEN
      DO;
         ak = ABS(t1) * 0.50000000000000000E0;
         flam = ak + SQRT(ak*ak - 1.00000000000000000E0);
         rho = MIN(ap2/ap1, flam);
         test = test1 * SQRT(rho/(rho*rho - 1.00000000000000000E0));
         itime = 2;
         GO TO L10;
      END;
   kk = k + 1 - id;
   ak = kk;
   dfnu = fnu + (n-1);
   cdfnu = dfnu;
   t1 = ak;
   p1 = 1.00000000000000000E0/ap2;
   p2 = czero;
   DO  i = 1 TO  kk;
      pt = p1;
      p1 = rz * (cdfnu+t1) * p1 + p2;
      p2 = pt;
      t1 = t1 - cone;
   END;
   IF  p1 = 0+0i  THEN p1 = COMPLEX(tol, tol);
   cy(n) = p2 / p1;
   IF  n  = 1 THEN RETURN;
   k = n - 1;
   ak = k;
   t1 = ak;
   cdfnu = fnu * rz;
   DO  i = 2 TO  n;
      pt = cdfnu + t1 * rz + cy(k+1);
      IF pt = 0+0i THEN pt = COMPLEX(tol, tol);
      cy(k) = cone / pt;
      t1 = t1 - cone;
      k = k - 1;
   END;
   RETURN;
   END crati;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbknu: PROCEDURE (z, fnu, kode, n, y, nz, tol, elim, alim) OPTIONS (REORDER);   
/* ***BEGIN PROLOGUE  CBKNU                                              */
/* ***REFER TO  CBESI,CBESK,CAIRY,CBESH                                  */

/*     CBKNU COMPUTES THE K BESSEL FUNCTION IN THE RIGHT HALF Z PLANE    */

/* ***ROUTINES CALLED  CKSCL,CSHCH,GAMLN,I1MACH,R1MACH,CUCHK             */
/* ***END PROLOGUE  CBKNU                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( cch, ck, coef, crsc, cs, cscl, csh, csr(3), css(3), cz, f, fmu, 
      p, pt, p1, p2, q, rz, smu, st, s1, s2, zd, celm, cy(2) )  COMPLEX FLOAT (18);
   DECLARE ( aa, ak, ascle, a1, a2, bb, bk, bry(3), caz, dnu, dnu2, etest, 
      fc, fhs, fk, fks, g1, g2, p2i, p2m, p2r, rk, s, tm, t1, t2, xx, yy, helim, 
      elm, xd, yd, alas, as )  FLOAT (18);
   DECLARE ( i, iflag, inu, k, kflag, kk, koded, nw, j, ic, inub )  FIXED 
      BINARY (31);

   DECLARE ( kmax  STATIC INITIAL ( 30)
               )  FIXED BINARY (31);
   DECLARE ( r1  STATIC INITIAL ( 2.00000000000000000E0)
               )  FLOAT (18);
   DECLARE ( czero  STATIC INITIAL
               (0.00000000000000000E0+0.00000000000000000E0i),
             cone   STATIC INITIAL
               (1.00000000000000000E0+0.00000000000000000E0i),
             ctwo   STATIC INITIAL
               (2.00000000000000000E0+0.00000000000000000E0i) ) COMPLEX FLOAT (18);

   DECLARE ( pi    STATIC INITIAL ( 3.14159265358979324E0)
           , rthpi STATIC INITIAL ( 1.25331413731550025E0)
           , spi   STATIC INITIAL ( 1.90985931710274403E0)
           , hpi   STATIC INITIAL ( 1.57079632679489662E0)
           , fpi   STATIC INITIAL ( 1.89769999331517738E0)
           , tth   STATIC INITIAL ( 6.66666666666666666E-01) )  FLOAT (18);

   DECLARE ( cc(8)  STATIC INITIAL (
                    5.77215664901532861E-01,
                   -4.20026350340952355E-02,
                   -4.21977345555443367E-02,
                    7.21894324666309954E-03,
                   -2.15241674114950973E-04,
                   -2.01348547807882387E-05,
                    1.13302723198169588E-06,
                    6.11609510448141582E-09 )  )
                FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, iflag, inu, k, kflag, kk, koded, nw, j, ic, inub );
      RESIGNAL;
   END;

   xx = REAL(z);
   yy = IMAG(z);
   caz = ABS(z);
   cscl = 1.00000000000000000E0/tol;
   crsc = tol;
   css(1) = cscl;
   css(2) = cone;
   css(3) = crsc;
   csr(1) = crsc;
   csr(2) = cone;
   csr(3) = cscl;
   bry(1) = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
   bry(2) = 1.00000000000000000E0 / bry(1);
   bry(3) = HUGE(0.00000000000000000E0);
   nz = 0;
   iflag = 0;
   koded = kode;
   rz = ctwo / z;
   inu = fnu + 0.50000000000000000E0;
   dnu = fnu - inu;
   IF  ABS(dnu) ^= 0.50000000000000000E0  THEN
      DO;
         dnu2 = 0;
         IF  ABS(dnu) > tol THEN
            dnu2 = dnu * dnu;
         IF  caz <= r1  THEN
            DO;
/* ----------------------------------------------------------------------- */
/*     SERIES FOR ABS(Z) <= R1                                           */
/* ----------------------------------------------------------------------- */
               fc = 1;
               smu = LOG(rz);
               fmu = smu * dnu;
               CALL cshch(fmu, csh, cch);
               IF  dnu ^= 0  THEN
                  DO;
                     fc = dnu * pi;
                     fc = fc / SIN(fc);
                     smu = csh / dnu;
                  END;
               a2 = 1.00000000000000000E0 + dnu;
/* ----------------------------------------------------------------------- */
/*     GAM(1-Z)*GAM(1+Z)=PI*Z/SIN(PI*Z), T1=1/GAM(1-DNU), T2=1/GAM(1+DNU) */
/* ----------------------------------------------------------------------- */
               t2 = EXP(-gamln(a2));
               t1 = 1.00000000000000000E0 / (t2*fc);
               IF  ABS(dnu) <= 0.10000000000000000E0  THEN
                  DO;
/* ----------------------------------------------------------------------- */
/*     SERIES FOR F0 TO RESOLVE INDETERMINACY FOR SMALL ABS(DNU)         */
/* ----------------------------------------------------------------------- */
                     ak = 1;
                     s = cc(1);
                     DO  k = 2 TO  8;
                        ak = ak * dnu2;
                        tm = cc(k) * ak;
                        s = s + tm;
                        IF  ABS(tm) < tol THEN
                           LEAVE;
                     END;
                     g1 = -s;
                  END;
               ELSE
                  DO;
                     g1 = (t1-t2) / (dnu+dnu);
                  END;
               g2 = 0.50000000000000000E0 * (t1+t2) * fc;
               g1 = g1 * fc;
               f = g1 * cch + smu * g2;
               pt = EXP(fmu);
               p = COMPLEX(0.50000000000000000E0/t2, 0) * pt;
               q = COMPLEX(0.50000000000000000E0/t1, 0) / pt;
               s1 = f;
               s2 = p;
               ak = 1.00000000000000000E0;
               a1 = 1.00000000000000000E0;
               ck = cone;
               bk = 1.00000000000000000E0 - dnu2;
               IF  inu <= 0   &   n <= 1  THEN
                  DO;
/* ----------------------------------------------------------------------- */
/*     GENERATE K(FNU,Z), 0.0E0  <=  FNU  <  0.5E0 AND N=1               */
/* ----------------------------------------------------------------------- */
                     IF  caz >= tol  THEN
                        DO;
                           cz = z * z * 0.25000000000000000E0;
                           t1 = 0.25000000000000000E0 * caz * caz;

L30:
                           f = (f*ak + p + q) / bk;
                           p = p / (ak-dnu);
                           q = q / (ak+dnu);
                           rk = 1.00000000000000000E0 / ak;
                           ck = ck * cz * rk;
                           s1 = s1 + ck * f;
                           a1 = a1 * t1 * rk;
                           bk = bk + ak + ak + 1.00000000000000000E0;
                           ak = ak + 1.00000000000000000E0;
                           IF  a1 > tol THEN  GO TO L30;
                        END;
                     y(1) = s1;
                     IF  koded  = 1 THEN
                        RETURN;
                     y(1) = s1 * EXP(z);
                     RETURN;
                  END;
/* ----------------------------------------------------------------------- */
/*     GENERATE K(DNU,Z) AND K(DNU+1,Z) FOR FORWARD RECURRENCE           */
/* ----------------------------------------------------------------------- */
               IF  caz >= tol  THEN
                  DO;
                     cz = z * z * 0.25000000000000000E0;
                     t1 = 0.25000000000000000E0 * caz * caz;

L40:
                     f = (f*ak + p + q) / bk;
                     p = p / (ak-dnu);
                     q = q / (ak+dnu);
                     rk = 1.00000000000000000E0 / ak;
                     ck = ck * cz * rk;
                     s1 = s1 + ck * f;
                     s2 = s2 + ck * (p - f*ak);
                     a1 = a1 * t1 * rk;
                     bk = bk + ak + ak + 1.00000000000000000E0;
                     ak = ak + 1.00000000000000000E0;
                     IF  a1 > tol THEN  GO TO L40;
                  END;
               kflag = 2;
               bk = REAL(smu);
               a1 = fnu + 1.00000000000000000E0;
               ak = a1 * ABS(bk);
               IF  ak > alim THEN
                  kflag = 3;
               p2 = s2 * css(kflag);
               s2 = p2 * rz;
               s1 = s1 * css(kflag);
               IF  koded  = 1 THEN  GO TO L100;
               f = EXP(z);
               s1 = s1 * f;
               s2 = s2 * f;
               GO TO L100;
            END;
      END;
/* ----------------------------------------------------------------------- */
/*     IFLAG=0 MEANS NO UNDERFLOW OCCURRED                               */
/*     IFLAG=1 MEANS AN UNDERFLOW OCCURRED- COMPUTATION PROCEEDS WITH    */
/*     KODED=2 AND A TEST FOR ON SCALE VALUES IS MADE DURING FORWARD     */
/*     RECURSION                                                         */
/* ----------------------------------------------------------------------- */
   coef = rthpi / SQRT(z);
   kflag = 2;
   IF  koded ^= 2  THEN
      DO;
         IF  xx > alim THEN  GO TO L200;
         a1 = EXP(-xx) * REAL(css(kflag) );
         pt = a1 * COMPLEX(COS(yy), -SIN(yy) );
         coef = coef * pt;
      END;

L50:
   IF  ABS(dnu)  = 0.50000000000000000E0 THEN  GO TO L210;
/* ----------------------------------------------------------------------- */
/*     MILLER ALGORITHM FOR ABS(Z) > R1                                  */
/* ----------------------------------------------------------------------- */
   ak = COS(pi*dnu);
   ak = ABS(ak);
   IF  ak  = 0.00000000000000000E0 THEN  GO TO L210;
   fhs = ABS(0.25000000000000000E0 - dnu2);
   IF  fhs  = 0.00000000000000000E0 THEN  GO TO L210;
/* ----------------------------------------------------------------------- */
/*     COMPUTE R2=F(E). IF ABS(Z) >= R2, USE FORWARD RECURRENCE TO       */
/*     DETERMINE THE BACKWARD INDEX K. R2=F(E) IS A STRAIGHT LINE ON     */
/*     12 <= E <= 60. E IS COMPUTED FROM 2**(-E)=B**(1-DIGITS(0.0))=     */
/*     TOL WHERE B IS THE BASE OF THE ARITHMETIC.                        */
/* ----------------------------------------------------------------------- */
   t1 = (DIGITS(0.00000000000000000E0) - 1) *
         LOG10( RADIX(0.00000000000000000E0) ) * 3.32192809400000000E0;
   t1 = MAX(t1,12.0000000000000000E0);
   t1 = MIN(t1,60.0000000000000000E0);
   t2 = tth * t1 - 6.00000000000000000E0;
   IF  xx  = 0.00000000000000000E0  THEN
      DO;
         t1 = hpi;
      END;
   ELSE
      DO;
         t1 = ATAN(yy/xx);
         t1 = ABS(t1);
      END;
   IF  t2 <= caz  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     FORWARD RECURRENCE LOOP WHEN ABS(Z) >= R2                         */
/* ----------------------------------------------------------------------- */
         etest = ak / (pi*caz*tol);
         fk = 1.00000000000000000E0;
         IF  etest < 1 THEN  GO TO L80;
         fks = 2;
         rk = caz + caz + 2;
         a1 = 0.00000000000000000E0;
         a2 = 1.00000000000000000E0;
         DO  i = 1 TO  kmax;
            ak = fhs / fks;
            bk = rk / (fk+1.00000000000000000E0);
            tm = a2;
            a2 = bk * a2 - ak * a1;
            a1 = tm;
            rk = rk + 2.00000000000000000E0;
            fks = fks + fk + fk + 2.00000000000000000E0;
            fhs = fhs + fk + fk;
            fk = fk + 1.00000000000000000E0;
            tm = ABS(a2) * fk;
            IF  etest < tm THEN  GO TO L70;
         END;
         GO TO L220;

L70:
         fk = fk + spi * t1 * SQRT(t2/caz);
         fhs = ABS(0.25000000000000000E0-dnu2);
      END;
   ELSE
      DO;
/* ----------------------------------------------------------------------- */
/*     COMPUTE BACKWARD INDEX K FOR ABS(Z) < R2                          */
/* ----------------------------------------------------------------------- */
         a2 = SQRT(caz);
         ak = fpi * ak / (tol*SQRT(a2));
         aa = 3.00000000000000000E0 * t1 / (1.00000000000000000E0+caz);
         bb = 14.7000000000000000E0 * t1 / (28.0000000000000000E0+caz);
         ak = (LOG(ak) + caz*COS(aa)/(1 + 0.00800000000000000E0*caz)) / COS(bb);
         fk = 0.12125000000000000E0 * ak * ak / caz + 1.50000000000000000E0;
      END;

L80:
   k = fk;
/* ----------------------------------------------------------------------- */
/*     BACKWARD RECURRENCE LOOP FOR MILLER ALGORITHM                     */
/* ----------------------------------------------------------------------- */
   fk = k;
   fks = fk * fk;
   p1 = czero;
   p2 = tol;
   cs = p2;
   DO  i = 1 TO  k;
      a1 = fks - fk;
      a2 = (fks+fk) / (a1+fhs);
      rk = 2 / (fk + 1.00000000000000000E0);
      t1 = (fk+xx) * rk;
      t2 = yy * rk;
      pt = p2;
      p2 = (p2*COMPLEX(t1, t2) - p1) * a2;
      p1 = pt;
      cs = cs + p2;
      fks = a1 - fk + 1.00000000000000000E0;
      fk = fk - 1.00000000000000000E0;
   END;
/* ----------------------------------------------------------------------- */
/*     COMPUTE (P2/CS)=(P2/ABS(CS))*(CONJG(CS)/ABS(CS)) FOR BETTER SCALING */
/* ----------------------------------------------------------------------- */
   tm = ABS(cs);
   pt = COMPLEX(1/tm, 0);
   s1 = pt * p2;
   cs = CONJG(cs) * pt;
   s1 = coef * s1 * cs;
   IF  inu <= 0   &   n <= 1  THEN
      DO;
         zd = z;
         IF  iflag  = 1 THEN  GO TO L190;
         GO TO L130;
      END;
/* ----------------------------------------------------------------------- */
/*     COMPUTE P1/P2=(P1/ABS(P2)*CONJG(P2)/ABS(P2) FOR SCALING           */
/* ----------------------------------------------------------------------- */
   tm = ABS(p2);
   pt = COMPLEX(1/tm, 0);
   p1 = pt * p1;
   p2 = CONJG(p2) * pt;
   pt = p1 * p2;
   s2 = s1 * (cone + (COMPLEX(dnu+0.50000000000000000E0, 0) - pt)/z);
/* ----------------------------------------------------------------------- */
/*     FORWARD RECURSION ON THE THREE TERM RECURSION RELATION WITH       */
/*     SCALING NEAR EXPONENT EXTREMES ON KFLAG=1 OR KFLAG=3              */
/* ----------------------------------------------------------------------- */
L100:
   ck = COMPLEX(dnu+1, 0) * rz;
   IF  n  = 1 THEN
      inu = inu - 1;
   IF  inu <= 0  THEN
      DO;
         IF  n  = 1 THEN
            s1 = s2;
         zd = z;
         IF  iflag  = 1 THEN  GO TO L190;
         GO TO L130;
      END;
   inub = 1;
   IF  iflag  = 1 THEN  GO TO L160;

L110:
   p1 = csr(kflag);
   ascle = bry(kflag);
   DO  i = inub TO  inu;
      st = s2;
      s2 = ck * s2 + s1;
      s1 = st;
      ck = ck + rz;
      IF  kflag < 3  THEN
         DO;
            p2 = s2 * p1;
            p2r = REAL(p2);
            p2i = IMAG(p2);
            p2r = ABS(p2r);
            p2i = ABS(p2i);
            p2m = MAX(p2r,p2i);
            IF  p2m > ascle  THEN
               DO;
                  kflag = kflag + 1;
                  ascle = bry(kflag);
                  s1 = s1 * p1;
                  s2 = p2;
                  s1 = s1 * css(kflag);
                  s2 = s2 * css(kflag);
                  p1 = csr(kflag);
               END;
         END;
   END;
   IF  n  = 1 THEN
      s1 = s2;

L130:
   y(1) = s1 * csr(kflag);
   IF  n  = 1 THEN
      RETURN;
   y(2) = s2 * csr(kflag);
   IF  n  = 2 THEN
      RETURN;
   kk = 2;

L140:
   kk = kk + 1;
   IF  kk > n THEN
      RETURN;
   p1 = csr(kflag);
   ascle = bry(kflag);
   DO  i = kk TO  n;
      p2 = s2;
      s2 = ck * s2 + s1;
      s1 = p2;
      ck = ck + rz;
      p2 = s2 * p1;
      y(i) = p2;
      IF  kflag < 3  THEN
         DO;
            p2r = REAL(p2);
            p2i = IMAG(p2);
            p2r = ABS(p2r);
            p2i = ABS(p2i);
            p2m = MAX(p2r,p2i);
            IF  p2m > ascle  THEN
               DO;
                  kflag = kflag + 1;
                  ascle = bry(kflag);
                  s1 = s1 * p1;
                  s2 = p2;
                  s1 = s1 * css(kflag);
                  s2 = s2 * css(kflag);
                  p1 = csr(kflag);
               END;
         END;
   END;
   RETURN;
/* ----------------------------------------------------------------------- */
/*     IFLAG=1 CASES, FORWARD RECURRENCE ON SCALED VALUES ON UNDERFLOW   */
/* ----------------------------------------------------------------------- */
L160:
   helim = 0.50000000000000000E0 * elim;
   elm = EXP(-elim);
   celm = elm;
   ascle = bry(1);
   zd = z;
   xd = xx;
   yd = yy;
   ic = -1;
   j = 2;

L165:
   DO  i = 1 TO  inu;
      st = s2;
      s2 = ck * s2 + s1;
      s1 = st;
      ck = ck + rz;
      as = ABS(s2);
      alas = LOG(as);
      p2r = -xd + alas;
      IF  p2r >= -elim  THEN
         DO;
            p2 = -zd + LOG(s2);
            p2r = REAL(p2);
            p2i = IMAG(p2);
            p2m = EXP(p2r) / tol;
            p1 = p2m * COMPLEX(COS(p2i), SIN(p2i) );
            CALL cuchk(p1, nw, ascle, tol);
            IF  nw  = 0  THEN
               DO;
                  j = 3 - j;
                  cy(j) = p1;
                  IF  ic  = i-1 THEN  GO TO L180;
                  ic = i;
                  ITERATE L165;
               END;
         END;
      IF  alas >= helim  THEN
         DO;
            xd = xd - elim;
            s1 = s1 * celm;
            s2 = s2 * celm;
            zd = COMPLEX(xd, yd);
         END;
   END;
   IF  n  = 1 THEN
      s1 = s2;
   GO TO L190;

L180:
   kflag = 1;
   inub = i + 1;
   s2 = cy(j);
   j = 3 - j;
   s1 = cy(j);
   IF  inub <= inu THEN  GO TO L110;
   IF  n  = 1 THEN
      s1 = s2;
   GO TO L130;

L190:
   y(1) = s1;
   IF  n ^= 1  THEN
      DO;
         y(2) = s2;
      END;
   ascle = bry(1);
   CALL ckscl(zd, fnu, n, y, nz, rz, ascle, tol, elim);
   inu = n - nz;
   IF  inu <= 0 THEN
      RETURN;
   kk = nz + 1;
   s1 = y(kk);
   y(kk) = s1 * csr(1);
   IF  inu  = 1 THEN
      RETURN;
   kk = nz + 2;
   s2 = y(kk);
   y(kk) = s2 * csr(1);
   IF  inu  = 2 THEN
      RETURN;
   t2 = fnu + (kk-1);
   ck = t2 * rz;
   kflag = 1;
   GO TO L140;
/* ----------------------------------------------------------------------- */
/*     SCALE BY EXP(Z), IFLAG = 1 CASES                                  */
/* ----------------------------------------------------------------------- */
L200:
   koded = 2;
   iflag = 1;
   kflag = 2;
   GO TO L50;
/* ----------------------------------------------------------------------- */
/*     FNU=HALF ODD INTEGER CASE, DNU=-0.5                               */
/* ----------------------------------------------------------------------- */
L210:
   s1 = coef;
   s2 = coef;
   GO TO L100;

L220:
   nz = -2;
   RETURN;
   END cbknu;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
ckscl: PROCEDURE (zr, fnu, n, y, nz, rz, ascle, tol, elim) OPTIONS (REORDER);   
/* ***BEGIN PROLOGUE  CKSCL                                              */
/* ***REFER TO  CBKNU,CUNK1,CUNK2                                        */

/*     SET K FUNCTIONS TO ZERO ON UNDERFLOW, CONTINUE RECURRENCE         */
/*     ON SCALED FUNCTIONS UNTIL TWO MEMBERS COME ON SCALE, THEN         */
/*     RETURN WITH MIN(NZ+2,N) VALUES SCALED BY 1/TOL.                   */

/* ***ROUTINES CALLED  CUCHK                                             */
/* ***END PROLOGUE  CKSCL                                                */

   DECLARE ( zr )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( rz )  COMPLEX FLOAT (18);
   DECLARE ( ascle )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);

   DECLARE ( ck, cs, cy(2), s1, s2, zd, celm )  COMPLEX FLOAT (18);
   DECLARE ( aa, acs, as, csi, csr, fn, xx, zri, elm, alas, helim )  FLOAT (18);
   DECLARE ( i, ic, kk, nn, nw )  FIXED BINARY (31);
   DECLARE czero  STATIC INITIAL (0+0i)  COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, ic, kk, nn, nw );
      RESIGNAL;
   END;

   nz = 0;
   ic = 0;
   xx = REAL(zr);
   nn = MIN(2,n);
   DO  i = 1 TO  nn;
      s1 = y(i);
      cy(i) = s1;
      as = ABS(s1);
      acs = -xx + LOG(as);
      nz = nz + 1;
      y(i) = czero;
      IF  acs >= -elim  THEN
         DO;
            cs = -zr + LOG(s1);
            csr = REAL(cs);
            csi =  IMAG(cs);
            aa = EXP(csr) / tol;
            cs = aa * COMPLEX(COS(csi), SIN(csi));
            CALL cuchk(cs, nw, ascle, tol);
            IF  nw  = 0  THEN
               DO;
                  y(i) = cs;
                  nz = nz - 1;
                  ic = i;
               END;
         END;
   END;
   IF  n  = 1 THEN
      RETURN;
   IF  ic <= 1  THEN
      DO;
         y(1) = czero;
         nz = 2;
      END;
   IF  n  = 2 THEN
      RETURN;
   IF  nz  = 0 THEN
      RETURN;
   fn = fnu + 1.00000000000000000E0;
   ck = fn * rz;
   s1 = cy(1);
   s2 = cy(2);
   helim = 0.50000000000000000E0 * elim;
   elm = EXP(-elim);
   celm = elm;
   zri =  IMAG(zr);
   zd = zr;

/*     FIND TWO CONSECUTIVE Y VALUES ON SCALE. SCALE RECURRENCE IF       */
/*     S2 GETS LARGER THAN EXP(ELIM/2)                                   */

L05:
   DO  i = 3 TO  n;
      kk = i;
      cs = s2;
      s2 = ck * s2 + s1;
      s1 = cs;
      ck = ck + rz;
      as = ABS(s2);
      alas = LOG(as);
      acs = -xx + alas;
      nz = nz + 1;
      y(i) = czero;
      IF  acs >= -elim  THEN
         DO;
            cs = -zd + LOG(s2);
            csr = REAL(cs);
            csi = IMAG(cs);
            aa = EXP(csr) / tol;
            cs = aa * COMPLEX(COS(csi), SIN(csi) );
            CALL cuchk(cs, nw, ascle, tol);
            IF  nw  = 0  THEN
               DO;
                  y(i) = cs;
                  nz = nz - 1;
                  IF  ic  = kk-1 THEN  GO TO L30;
                  ic = kk;
                  ITERATE L05;
               END;
         END;
      IF  alas >= helim  THEN
         DO;
            xx = xx - elim;
            s1 = s1 * celm;
            s2 = s2 * celm;
            zd = COMPLEX(xx, zri);
         END;
   END;
   nz = n;
   IF  ic  = n THEN
      nz = n - 1;
   GO TO L40;

L30:
   nz = kk - 2;

L40:
   DO i = 1 TO nz; y(i) = czero; END;
   RETURN;
   END ckscl;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cacon: PROCEDURE (z, fnu, kode, mr, n, y, nz, rl, fnul, tol, elim, alim)
          OPTIONS (REORDER);                                                                     
/* ***BEGIN PROLOGUE  CACON                                              */
/* ***REFER TO  CBESK,CBESH                                              */

/*     CACON APPLIES THE ANALYTIC CONTINUATION FORMULA                   */

/*         K(FNU,ZN*EXP(MP))=K(FNU,ZN)*EXP(-MP*FNU) - MP*I(FNU,ZN)       */
/*                 MP=PI*MR*CMPLX(0.0,1.0)                               */

/*     TO CONTINUE THE K FUNCTION FROM THE RIGHT HALF TO THE LEFT        */
/*     HALF Z PLANE                                                      */

/* ***ROUTINES CALLED  CBINU,CBKNU,CS1S2,R1MACH                          */
/* ***END PROLOGUE  CACON                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( mr )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( rl )  FLOAT (18);
   DECLARE ( fnul )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( ck, cs, cscl, cscr, csgn, cspn, css(3), csr(3), c1, c2, rz, sc1, 
      sc2, st, s1, s2, zn, cy(2) )  COMPLEX FLOAT (18);
   DECLARE ( arg, ascle, as2, bscle, bry(3), cpn, c1i, c1m, c1r, fmr, sgn, 
      spn, yy )  FLOAT (18);
   DECLARE ( i, inu, iuf, kflag, nn, nw )  FIXED BINARY (31);
   DECLARE ( pi  STATIC INITIAL ( 3.14159265358979324E0) )  FLOAT (18);
   DECLARE cone  STATIC INITIAL ( 1.00000000000000000E0+0.00000000000000000E0i)
      COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, inu, iuf, kflag, nn, nw );
      RESIGNAL;
   END;

   nz = 0;
   zn = -z;
   nn = n;
   CALL cbinu(zn, fnu, kode, nn, y, nw, rl, fnul, tol, elim, alim);
   IF  nw >= 0  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     ANALYTIC CONTINUATION TO THE LEFT HALF PLANE FOR THE K FUNCTION   */
/* ----------------------------------------------------------------------- */
         nn = MIN(2, n);
         CALL cbknu(zn, fnu, kode, nn, cy, nw, tol, elim, alim);
         IF  nw  = 0  THEN
            DO;
               s1 = cy(1);
               fmr = mr;
               sgn = -SIGN(pi, fmr);
               csgn = COMPLEX(0, sgn);
               IF  kode ^= 1  THEN
                  DO;
                     yy = - IMAG(zn);
                     cpn = COS(yy);
                     spn = SIN(yy);
                     csgn = csgn * COMPLEX(cpn, spn);
                  END;
/* ----------------------------------------------------------------------- */
/*     CALCULATE CSPN=EXP(FNU*PI*I) TO MINIMIZE LOSSES OF SIGNIFICANCE   */
/*     WHEN FNU IS LARGE                                                 */
/* ----------------------------------------------------------------------- */
               inu = fnu;
               arg = (fnu - inu) * sgn;
               cpn = COS(arg);
               spn = SIN(arg);
               cspn = COMPLEX(cpn, spn);
               IF  REM(inu, 2)  = 1 THEN
                  cspn = -cspn;
               iuf = 0;
               c1 = s1;
               c2 = y(1);
               ascle = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
               IF  kode ^= 1  THEN
                  DO;
                     CALL cs1s2(zn, c1, c2, nw, ascle, alim, iuf);
                     nz = nz + nw;
                     sc1 = c1;
                  END;
               y(1) = cspn * c1 + csgn * c2;
               IF  n  = 1 THEN
                  RETURN;
               cspn = -cspn;
               s2 = cy(2);
               c1 = s2;
               c2 = y(2);
               IF  kode ^= 1  THEN
                  DO;
                     CALL cs1s2(zn, c1, c2, nw, ascle, alim, iuf);
                     nz = nz + nw;
                     sc2 = c1;
                  END;
               y(2) = cspn * c1 + csgn * c2;
               IF  n  = 2 THEN
                  RETURN;
               cspn = -cspn;
               rz = 2.00000000000000000E0 / zn;
               ck = COMPLEX(fnu+1.00000000000000000E0, 0) * rz;
/* ----------------------------------------------------------------------- */
/*     SCALE NEAR EXPONENT EXTREMES DURING RECURRENCE ON K FUNCTIONS     */
/* ----------------------------------------------------------------------- */
               cscl = 1.00000000000000000E0/tol;
               cscr = tol;
               css(1) = cscl;
               css(2) = cone;
               css(3) = cscr;
               csr(1) = cscr;
               csr(2) = cone;
               csr(3) = cscl;
               bry(1) = ascle;
               bry(2) = 1.00000000000000000E0 / ascle;
               bry(3) = HUGE(0.00000000000000000E0);
               as2 = ABS(s2);
               kflag = 2;
               IF  as2 <= bry(1)  THEN
                  DO;
                     kflag = 1;
                  END;
               ELSE
                  DO;
                     IF  as2 >= bry(2)  THEN
                        DO;
                           kflag = 3;
                        END;
                  END;
               bscle = bry(kflag);
               s1 = s1 * css(kflag);
               s2 = s2 * css(kflag);
               cs = csr(kflag);
               DO  i = 3 TO  n;
                  st = s2;
                  s2 = ck * s2 + s1;
                  s1 = st;
                  c1 = s2 * cs;
                  st = c1;
                  c2 = y(i);
                  IF  kode ^= 1  THEN
                     DO;
                        IF  iuf >= 0  THEN
                           DO;
                              CALL cs1s2(zn, c1, c2, nw, ascle, alim, iuf);
                              nz = nz + nw;
                              sc1 = sc2;
                              sc2 = c1;
                              IF  iuf  = 3  THEN
                                 DO;
                                    iuf = -4;
                                    s1 = sc1 * css(kflag);
                                    s2 = sc2 * css(kflag);
                                    st = sc2;
                                 END;
                           END;
                     END;
                  y(i) = cspn * c1 + csgn * c2;
                  ck = ck + rz;
                  cspn = -cspn;
                  IF  kflag < 3  THEN
                     DO;
                        c1r = REAL(c1);
                        c1i = IMAG(c1);
                        c1r = ABS(c1r);
                        c1i = ABS(c1i);
                        c1m = MAX(c1r, c1i);
                        IF  c1m > bscle  THEN
                           DO;
                              kflag = kflag + 1;
                              bscle = bry(kflag);
                              s1 = s1 * cs;
                              s2 = st;
                              s1 = s1 * css(kflag);
                              s2 = s2 * css(kflag);
                              cs = csr(kflag);
                           END;
                     END;
               END;
               RETURN;
            END;
      END;
   nz = -1;
   IF  nw  = -2 THEN
      nz = -2;
   RETURN;

DECLARE SIGN GENERIC (DSIGN WHEN (FLOAT, FLOAT),
   ISIGN WHEN (FIXED BINARY, FIXED BINARY) );
ISIGN: PROCEDURE (X, Y) RETURNS (FIXED BINARY(31)) OPTIONS (INLINE);
   DECLARE (X, Y) FIXED BINARY (31);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END ISIGN;
DSIGN: PROCEDURE (X, Y) RETURNS (FLOAT (18)) OPTIONS (INLINE);
   DECLARE (X, Y) FLOAT (18);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END DSIGN;
   END cacon;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cbinu: PROCEDURE (z, fnu, kode, n, cy, nz, rl, fnul, tol, elim, alim)
          OPTIONS (REORDER);                                                                        
/* ***BEGIN PROLOGUE  CBINU                                              */
/* ***REFER TO  CBESH,CBESI,CBESJ,CBESK,CAIRY,CBIRY                      */

/*     CBINU COMPUTES THE I FUNCTION IN THE RIGHT HALF Z PLANE           */

/* ***ROUTINES CALLED  CASYI,CBUNI,CMLRI,CSERI,CUOIK,CWRSK               */
/* ***END PROLOGUE  CBINU                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( cy(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( rl )  FLOAT (18);
   DECLARE ( fnul )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( cw(2) )  COMPLEX FLOAT (18);
   DECLARE ( az, dfnu )  FLOAT (18);
   DECLARE ( inw, nlast, nn, nui, nw )  FIXED BINARY (31);
   DECLARE ( czero  STATIC INITIAL ( 0+0i ) ) COMPLEX FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( inw, nlast, nn, nui, nw );
      RESIGNAL;
   END;

   nz = 0;
   az = ABS(z);
   nn = n;
   cy = czero;
   dfnu = fnu + (n-1);
   IF  az > 2  THEN
      DO;
         IF  az*az*0.25000000000000000E0 > dfnu+1 THEN  GO TO L10;
      END;
/* ----------------------------------------------------------------------- */
/*     POWER SERIES                                                      */
/* ----------------------------------------------------------------------- */
   CALL cseri(z, fnu, kode, nn, cy, nw, tol, elim, alim);
   inw = ABS(nw);
   nz = nz + inw;
   nn = nn - inw;
   IF  nn  = 0 THEN
      RETURN;
   IF  nw >= 0 THEN  GO TO L80;
   dfnu = fnu + (nn-1);

L10:
   IF  az >= rl  THEN
      DO;
         IF  dfnu > 1  THEN
            DO;
               IF  az+az < dfnu*dfnu THEN  GO TO L20;
            END;
/* ----------------------------------------------------------------------- */
/*     ASYMPTOTIC EXPANSION FOR LARGE Z                                  */
/* ----------------------------------------------------------------------- */
         CALL casyi(z, fnu, kode, nn, cy, nw, rl, tol, elim, alim);
         IF  nw < 0 THEN  GO TO L90;
         GO TO L80;
      END;
   IF  dfnu <= 1.00000000000000000E0 THEN  GO TO L40;
/* ----------------------------------------------------------------------- */
/*     OVERFLOW AND UNDERFLOW TEST ON I SEQUENCE FOR MILLER ALGORITHM    */
/* ----------------------------------------------------------------------- */
L20:
   CALL cuoik(z, fnu, kode, 1, nn, cy, nw, tol, elim, alim);
   IF  nw < 0 THEN  GO TO L90;
   nz = nz + nw;
   nn = nn - nw;
   IF  nn  = 0 THEN
      RETURN;
   dfnu = fnu + (nn-1);
   IF  dfnu > fnul THEN  GO TO L70;
   IF  az > fnul THEN  GO TO L70;

L30:
   IF  az > rl THEN  GO TO L50;
/* ----------------------------------------------------------------------- */
/*     MILLER ALGORITHM NORMALIZED BY THE SERIES                         */
/* ----------------------------------------------------------------------- */
L40:
   CALL cmlri(z, fnu, kode, nn, cy, nw, tol);
   IF  nw < 0 THEN  GO TO L90;
   GO TO L80;
/* ----------------------------------------------------------------------- */
/*     MILLER ALGORITHM NORMALIZED BY THE WRONSKIAN                      */
/* ----------------------------------------------------------------------- */
/*     OVERFLOW TEST ON K FUNCTIONS USED IN WRONSKIAN                    */
/* ----------------------------------------------------------------------- */
L50:
   CALL cuoik(z, fnu, kode, 2, 2, cw, nw, tol, elim, alim);
   IF  nw < 0  THEN
      DO;
         nz = nn;
         DO nn = 1 TO nn;
            cy(nn) = czero;
         END;
         RETURN;
      END;
   IF  nw > 0 THEN  GO TO L90;
   CALL cwrsk(z, fnu, kode, nn, cy, nw, cw, tol, elim, alim);
   IF  nw < 0 THEN  GO TO L90;
   GO TO L80;
/* ----------------------------------------------------------------------- */
/*     INCREMENT FNU+NN-1 UP TO FNUL, COMPUTE AND RECUR BACKWARD         */
/* ----------------------------------------------------------------------- */
L70:
   nui = fnul - dfnu + 1;
   nui = MAX(nui, 0);
   CALL cbuni(z, fnu, kode, nn, cy, nw, nui, nlast, fnul, tol, elim, alim);
   IF  nw < 0 THEN  GO TO L90;
   nz = nz + nw;
   IF  nlast ^= 0  THEN
      DO;
         nn = nlast;
         GO TO L30;
      END;

L80:
   RETURN;

L90:
   nz = -1;
   IF  nw  = -2 THEN
      nz = -2;
   RETURN;
   END cbinu;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
gamln: PROCEDURE (z) RETURNS (FLOAT (18)) OPTIONS (REORDER);

/* N.B. Argument IERR has been removed.                                  */

/* ***BEGIN PROLOGUE  GAMLN                                              */
/* ***DATE WRITTEN   830501   (YYMMDD)                                   */
/* ***REVISION DATE  830501   (YYMMDD)                                   */
/* ***CATEGORY NO.  B5F                                                  */
/* ***KEYWORDS  GAMMA FUNCTION,LOGARITHM OF GAMMA FUNCTION               */
/* ***AUTHOR  AMOS, DONALD E., SANDIA NATIONAL LABORATORIES              */
/* ***PURPOSE  TO COMPUTE THE LOGARITHM OF THE GAMMA FUNCTION            */
/* ***DESCRIPTION                                                        */

/*   GAMLN COMPUTES THE NATURAL LOG OF THE GAMMA FUNCTION FOR Z > 0.     */
/*   THE ASYMPTOTIC EXPANSION IS USED TO GENERATE VALUES GREATER THAN ZMIN */
/*   WHICH ARE ADJUSTED BY THE RECURSION G(Z+1)=Z*G(Z) FOR Z <= ZMIN.    */
/*   THE FUNCTION WAS MADE AS PORTABLE AS POSSIBLE BY COMPUTIMG ZMIN FROM THE */
/*   NUMBER OF BASE 10 DIGITS IN A WORD,                                 */
/*   RLN = MAX(-LOG10(EPSILON(0.0)), 0.5E-18)                            */
/*   LIMITED TO 18 DIGITS OF (RELATIVE) ACCURACY.                        */

/*   SINCE INTEGER ARGUMENTS ARE COMMON, A TABLE LOOK UP ON 100          */
/*   VALUES IS USED FOR SPEED OF EXECUTION.                              */

/*  DESCRIPTION OF ARGUMENTS                                             */

/*      INPUT                                                            */
/*        Z      - REAL ARGUMENT, Z > 0.0                                */

/*      OUTPUT                                                           */
/*        GAMLN  - NATURAL LOG OF THE GAMMA FUNCTION AT Z                */
/*        IERR   - ERROR FLAG                                            */
/*                 IERR=0, NORMAL RETURN, COMPUTATION COMPLETED          */
/*                 IERR=1, Z <= 0.0,    NO COMPUTATION                   */

/* ***REFERENCES  COMPUTATION OF BESSEL FUNCTIONS OF COMPLEX ARGUMENT    */
/*                 BY D. E. AMOS, SAND83-0083, MAY 1983.                 */
/* ***ROUTINES CALLED  I1MACH,R1MACH                                     */
/* ***END PROLOGUE  GAMLN                                                */

   DECLARE ( z )  FLOAT (18);
   DECLARE ( fn_val )  FLOAT (18);

   DECLARE ( i, i1m, k, mz, nz )  FIXED BINARY (31);
   DECLARE ( fln, fz, rln, s, tlg, trm, tst, t1, wdtol, zdmy, zinc, zm, zmin, 
      zp, zsq )  FLOAT (18);

/*           LNGAMMA(N), N=1,100                                         */
   DECLARE ( gln(100)  STATIC INITIAL (
                    0.00000000000000000E0,
                    0.00000000000000000E0,
                    6.93147180559945309E-01,
                    1.79175946922805500E0,
                    3.17805383034794562E0,
                    4.78749174278204599E0,
                    6.57925121201010100E0,
                    8.52516136106541430E0,
                    1.06046029027452502E+01,
                    1.28018274800814696E+01,
                    1.51044125730755153E+01,
                    1.75023078458738858E+01,
                    1.99872144956618861E+01,
                    2.25521638531234229E+01,
                    2.51912211827386815E+01,
                    2.78992713838408916E+01,
                    3.06718601060806728E+01,
                    3.35050734501368889E+01,
                    3.63954452080330536E+01,
                    3.93398841871994940E+01,
                    4.23356164607534850E+01,
                    4.53801388984769080E+01,
                    4.84711813518352239E+01,
                    5.16066755677643736E+01,
                    5.47847293981123192E+01,
                    5.80036052229805199E+01,
                    6.12617017610020020E+01,
                    6.45575386270063311E+01,
                    6.78897431371815350E+01,
                    7.12570389671680090E+01,
                    7.46582363488301644E+01,
                    7.80922235533153106E+01,
                    8.15579594561150372E+01,
                    8.50544670175815174E+01,
                    8.85808275421976788E+01,
                    9.21361756036870925E+01,
                    9.57196945421432025E+01,
                    9.93306124547874269E+01,
                    1.02968198614513813E+02,
                    1.06631760260643459E+02,
                    1.10320639714757395E+02,
                    1.14034211781461703E+02,
                    1.17771881399745072E+02,
                    1.21533081515438634E+02,
                    1.25317271149356895E+02,
                    1.29123933639127215E+02,
                    1.32952575035616310E+02,
                    1.36802722637326368E+02,
                    1.40673923648234259E+02,
                    1.44565743946344886E+02,
                    1.48477766951773032E+02,
                    1.52409592584497358E+02,
                    1.56360836303078785E+02,
                    1.60331128216630907E+02,
                    1.64320112263195181E+02,
                    1.68327445448427652E+02,
                    1.72352797139162802E+02,
                    1.76395848406997352E+02,
                    1.80456291417543771E+02,
                    1.84533828861449491E+02,
                    1.88628173423671591E+02,
                    1.92739047287844902E+02,
                    1.96866181672889994E+02,
                    2.01009316399281527E+02,
                    2.05168199482641199E+02,
                    2.09342586752536836E+02,
                    2.13532241494563261E+02,
                    2.17736934113954227E+02,
                    2.21956441819130334E+02,
                    2.26190548323727593E+02,
                    2.30439043565776952E+02,
                    2.34701723442818268E+02,
                    2.38978389561834323E+02,
                    2.43268849002982714E+02,
                    2.47572914096186884E+02,
                    2.51890402209723194E+02,
                    2.56221135550009525E+02,
                    2.60564940971863209E+02,
                    2.64921649798552801E+02,
                    2.69291097651019823E+02,
                    2.73673124285693704E+02,
                    2.78067573440366143E+02,
                    2.82474292687630396E+02,
                    2.86893133295426994E+02,
                    2.91323950094270308E+02,
                    2.95766601350760624E+02,
                    3.00220948647014132E+02,
                    3.04686856765668715E+02,
                    3.09164193580146922E+02,
                    3.13652829949879062E+02,
                    3.18152639620209327E+02,
                    3.22663499126726177E+02,
                    3.27185287703775217E+02,
                    3.31717887196928473E+02,
                    3.36261181979198477E+02,
                    3.40815058870799018E+02,
                    3.45379407062266854E+02,
                    3.49954118040770237E+02,
                    3.54539085519440809E+02,
                    3.59134205369575399E+02 )  )
                FLOAT (18);

/*             COEFFICIENTS OF ASYMPTOTIC EXPANSION                      */
   DECLARE ( cf(22)  STATIC INITIAL (
                    8.33333333333333333E-02,
                   -2.77777777777777778E-03,
                    7.93650793650793651E-04,
                   -5.95238095238095238E-04,
                    8.41750841750841751E-04,
                   -1.91752691752691753E-03,
                    6.41025641025641026E-03,
                   -2.95506535947712418E-02,
                    1.79644372368830573E-01,
                   -1.39243221690590112E0,
                    1.34028640441683920E+01,
                   -1.56848284626002017E+02,
                    2.19310333333333333E+03,
                   -3.61087712537249894E+04,
                    6.91472268851313067E+05,
                   -1.52382215394074162E+07,
                    3.82900751391414141E+08,
                   -1.08822660357843911E+10,
                    3.47320283765002252E+11,
                   -1.23696021422692745E+13,
                    4.88788064793079335E+14,
                   -2.13203339609193739E+16 )  )
                FLOAT (18);

/*             LN(2*PI)                                                  */
   DECLARE ( con  STATIC INITIAL ( 1.83787706640934548E0) )  FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( i, i1m, k, mz, nz );
      RESIGNAL;
   END;

/****FIRST EXECUTABLE STATEMENT  GAMLN                                  */
   IF  z > 0  THEN
      DO;
         IF  z <= 101.000000000000000E0  THEN
            DO;
               nz = z;
               fz = z - nz;
               IF  fz <= 0  THEN
                  DO;
                     IF  nz <= 100  THEN
                        DO;
                           fn_val = gln(nz);
                           RETURN (fn_val);
                        END;
                  END;
            END;
         wdtol = EPSILON(0.00000000000000000E0);
         wdtol = MAX(wdtol, 0.50000000000000000E-18);
         i1m = DIGITS(0.00000000000000000E0);
         rln = LOG10( RADIX(0.00000000000000000E0) ) * i1m;
         fln = MIN(rln,20.0000000000000000E0);
         fln = MAX(fln,3.00000000000000000E0);
         fln = fln - 3.00000000000000000E0;
         zm = 1.80000000000000000E0 + 0.38750000000000000E0 * fln;
         mz = zm + 1;
         zmin = mz;
         zdmy = z;
         zinc = 0;
         IF  z < zmin  THEN
            DO;
               zinc = zmin - nz;
               zdmy = z + zinc;
            END;
         zp = 1 / zdmy;
         t1 = cf(1) * zp;
         s = t1;
         IF  zp >= wdtol  THEN
            DO;
               zsq = zp * zp;
               tst = t1 * wdtol;
               DO  k = 2 TO  22;
                  zp = zp * zsq;
                  trm = cf(k) * zp;
                  IF  ABS(trm) < tst THEN
                     LEAVE;
                  s = s + trm;
               END;
            END;

         IF  zinc  = 0  THEN
            DO;
               tlg = LOG(z);
               fn_val = z * (tlg-1) + 0.50000000000000000E0 * (con-tlg) + s;
               RETURN (fn_val);
            END;
         zp = 1;
         nz = zinc;
         DO  i = 1 TO  nz;
            zp = zp * (z + (i-1));
         END;
         tlg = LOG(zdmy);
         fn_val = zdmy * (tlg-1) - LOG(zp) + 0.50000000000000000E0 
            * (con-tlg) + s;
         RETURN (fn_val);
      END;

   PUT SKIP LIST ('** ERROR: Zero or -ve argument for function GAMLN **');
   RETURN (fn_val);
   END gamln;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cuchk: PROCEDURE (y, nz, ascle, tol) OPTIONS (REORDER);                         
/* ***BEGIN PROLOGUE  CUCHK                                              */
/* ***REFER TO CSERI,CUOIK,CUNK1,CUNK2,CUNI1,CUNI2,CKSCL                 */

/*   Y ENTERS AS A SCALED QUANTITY WHOSE MAGNITUDE IS GREATER THAN       */
/*   EXP(-ALIM) = ASCLE = 1.0E+3*TINY(0.0)/TOL.  THE TEST IS MADE TO SEE */
/*   IF THE MAGNITUDE OF THE REAL OR IMAGINARY PART WOULD UNDERFLOW WHEN Y IS */
/*   SCALED (BY TOL) TO ITS PROPER VALUE.  Y IS ACCEPTED IF THE UNDERFLOW IS AT */
/*   LEAST ONE PRECISION BELOW THE MAGNITUDE OF THE LARGEST COMPONENT; OTHERWISE */
/*   THE PHASE ANGLE DOES NOT HAVE ABSOLUTE ACCURACY AND AN UNDERFLOW IS ASSUMED. */

/* ***ROUTINES CALLED  (NONE)                                            */
/* ***END PROLOGUE  CUCHK                                                */

   DECLARE ( y )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( ascle )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);

   DECLARE ( ss, st, yr, yi )  FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( ss, st, yr, yi );
      RESIGNAL;
   END;

   nz = 0;
   yr = REAL(y);
   yi = IMAG(y);
   yr = ABS(yr);
   yi = ABS(yi);
   st = MIN(yr, yi);
   IF  st > ascle THEN
      RETURN;
   ss = MAX(yr, yi);
   st = st / tol;
   IF  ss < st THEN
      nz = 1;
   RETURN;
   END cuchk;



(SUBSCRIPTRANGE, FIXEDOVERFLOW, SIZE):
cacai: PROCEDURE (z, fnu, kode, mr, n, y, nz, rl, tol, elim, alim)
          OPTIONS (REORDER);                                                                           
/* ***BEGIN PROLOGUE  CACAI                                              */
/* ***REFER TO  CAIRY                                                    */

/*  CACAI APPLIES THE ANALYTIC CONTINUATION FORMULA                      */

/*      K(FNU,ZN*EXP(MP)) = K(FNU,ZN)*EXP(-MP*FNU) - MP*I(FNU,ZN)        */
/*              MP = PI*MR*CMPLX(0.0,1.0)                                */

/*  TO CONTINUE THE K FUNCTION FROM THE RIGHT HALF TO THE LEFT HALF Z PLANE */
/*  FOR USE WITH CAIRY WHERE FNU=1/3 OR 2/3 AND N=1.                     */
/*  CACAI IS THE SAME AS CACON WITH THE PARTS FOR LARGER ORDERS AND      */
/*  RECURRENCE REMOVED.  A RECURSIVE CALL TO CACON CAN RESULT IF CACON   */
/*  IS CALLED FROM CAIRY.                                                */

/* ***ROUTINES CALLED  CASYI,CBKNU,CMLRI,CSERI,CS1S2,R1MACH              */
/* ***END PROLOGUE  CACAI                                                */

   DECLARE ( z )  COMPLEX FLOAT (18);
   DECLARE ( fnu )  FLOAT (18);
   DECLARE ( kode )  FIXED BINARY (31);
   DECLARE ( mr )  FIXED BINARY (31);
   DECLARE ( n )  FIXED BINARY (31);
   DECLARE ( y(n) )  COMPLEX FLOAT (18);
   DECLARE ( nz )  FIXED BINARY (31);
   DECLARE ( rl )  FLOAT (18);
   DECLARE ( tol )  FLOAT (18);
   DECLARE ( elim )  FLOAT (18);
   DECLARE ( alim )  FLOAT (18);

   DECLARE ( csgn, cspn, c1, c2, zn, cy(2) )  COMPLEX FLOAT (18);
   DECLARE ( arg, ascle, az, cpn, dfnu, fmr, sgn, spn, yy )  FLOAT (18);
   DECLARE ( inu, iuf, nn, nw )  FIXED BINARY (31);
   DECLARE ( pi  STATIC INITIAL ( 3.14159265358979324E0) )  FLOAT (18);

   ON ERROR SNAP BEGIN;
      ON ERROR SNAP SYSTEM;
      PUT SKIP DATA ( inu, iuf, nn, nw );
      RESIGNAL;
   END;

   nz = 0;
   zn = -z;
   az = ABS(z);
   nn = n;
   dfnu = fnu + (n-1);
   IF  az > 2  THEN
      DO;
         IF  az*az*0.25000000000000000E0 > dfnu+1.00000000000000000E0 THEN  GO TO L10;
      END;
/* ----------------------------------------------------------------------- */
/*     POWER SERIES FOR THE I FUNCTION                                   */
/* ----------------------------------------------------------------------- */
   CALL cseri(zn, fnu, kode, nn, y, nw, tol, elim, alim);
   GO TO L20;

L10:
   IF  az >= rl  THEN
      DO;
/* ----------------------------------------------------------------------- */
/*     ASYMPTOTIC EXPANSION FOR LARGE Z FOR THE I FUNCTION               */
/* ----------------------------------------------------------------------- */
         CALL casyi(zn, fnu, kode, nn, y, nw, rl, tol, elim, alim);
         IF  nw < 0 THEN  GO TO L30;
      END;
   ELSE
      DO;
/* ----------------------------------------------------------------------- */
/*     MILLER ALGORITHM NORMALIZED BY THE SERIES FOR THE I FUNCTION      */
/* ----------------------------------------------------------------------- */
         CALL cmlri(zn, fnu, kode, nn, y, nw, tol);
         IF  nw < 0 THEN  GO TO L30;
      END;
/* ----------------------------------------------------------------------- */
/*     ANALYTIC CONTINUATION TO THE LEFT HALF PLANE FOR THE K FUNCTION   */
/* ----------------------------------------------------------------------- */
L20:
   CALL cbknu(zn, fnu, kode, 1, cy, nw, tol, elim, alim);
   IF  nw  = 0  THEN
      DO;
         fmr = mr;
         sgn = -SIGN(pi, fmr);
         csgn = COMPLEX(0, sgn);
         IF  kode ^= 1  THEN
            DO;
               yy = -IMAG(zn);
               cpn = COS(yy);
               spn = SIN(yy);
               csgn = csgn * COMPLEX(cpn, spn);
            END;
/* ----------------------------------------------------------------------- */
/*     CALCULATE CSPN=EXP(FNU*PI*I) TO MINIMIZE LOSSES OF SIGNIFICANCE   */
/*     WHEN FNU IS LARGE                                                 */
/* ----------------------------------------------------------------------- */
         inu = fnu;
         arg = (fnu - inu) * sgn;
         cpn = COS(arg);
         spn = SIN(arg);
         cspn = COMPLEX(cpn, spn);
         IF  REM(inu,2)  = 1 THEN
            cspn = -cspn;
         c1 = cy(1);
         c2 = y(1);
         IF  kode ^= 1  THEN
            DO;
               iuf = 0;
               ascle = 1.00000000000000000E+3 * TINY(0.00000000000000000E0) / tol;
               CALL cs1s2(zn, c1, c2, nw, ascle, alim, iuf);
               nz = nz + nw;
            END;
         y(1) = cspn * c1 + csgn * c2;
         RETURN;
      END;

L30:
   nz = -1;
   IF  nw  = -2 THEN
      nz = -2;
   RETURN;

DECLARE SIGN GENERIC (DSIGN WHEN (FLOAT, FLOAT),
   ISIGN WHEN (FIXED BINARY, FIXED BINARY) );
ISIGN: PROCEDURE (X, Y) RETURNS (FIXED BINARY(31)) OPTIONS (INLINE);
   DECLARE (X, Y) FIXED BINARY (31);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END ISIGN;
DSIGN: PROCEDURE (X, Y) RETURNS (FLOAT (18)) OPTIONS (INLINE);
   DECLARE (X, Y) FLOAT (18);
   IF Y < 0 THEN RETURN (-ABS(X)); ELSE RETURN (ABS(X));
END DSIGN;

   END cacai;

%INCLUDE 'MAX_EXP.INC';

